<?php
/**
 * Account Settings
 * OBGYN Research & Collaboration Platform
 */

require_once '../config/config.php';

requireLogin();
if (!isResearcher()) {
    redirect(SITE_URL . '/index.php');
}

$pageTitle = 'Settings';
$userId = getCurrentUserId();
$error = '';
$success = '';

// Get user data
$db = getDB();
$stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$userId]);
$user = $stmt->fetch();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    validateCSRF();
    
    $action = $_POST['action'] ?? '';
    
    if ($action === 'change_password') {
        $currentPassword = $_POST['current_password'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        
        if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
            $error = 'All password fields are required.';
        } elseif ($newPassword !== $confirmPassword) {
            $error = 'New passwords do not match.';
        } elseif (strlen($newPassword) < 8) {
            $error = 'Password must be at least 8 characters long.';
        } else {
            $result = changePassword($userId, $currentPassword, $newPassword);
            if ($result['success']) {
                $success = $result['message'];
            } else {
                $error = $result['message'];
            }
        }
        
    } elseif ($action === 'update_email') {
        $newEmail = sanitize($_POST['email'] ?? '');
        
        if (empty($newEmail)) {
            $error = 'Email is required.';
        } elseif (!filter_var($newEmail, FILTER_VALIDATE_EMAIL)) {
            $error = 'Invalid email format.';
        } elseif ($newEmail !== $user['email']) {
            // Check if email already exists
            $stmt = $db->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
            $stmt->execute([$newEmail, $userId]);
            if ($stmt->fetch()) {
                $error = 'Email already in use.';
            } else {
                $stmt = $db->prepare("UPDATE users SET email = ?, email_verified = 0 WHERE id = ?");
                $stmt->execute([$newEmail, $userId]);
                $success = 'Email updated. Please verify your new email address.';
                
                // Refresh user data
                $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
                $stmt->execute([$userId]);
                $user = $stmt->fetch();
            }
        }
        
    } elseif ($action === 'toggle_2fa') {
        $enable2FA = isset($_POST['enable_2fa']) ? 1 : 0;
        
        if ($enable2FA && empty($user['two_factor_secret'])) {
            // Generate 2FA secret
            $secret = generateToken(32);
            $stmt = $db->prepare("UPDATE users SET two_factor_enabled = 1, two_factor_secret = ? WHERE id = ?");
            $stmt->execute([$secret, $userId]);
            $success = 'Two-factor authentication enabled.';
        } else {
            $stmt = $db->prepare("UPDATE users SET two_factor_enabled = ? WHERE id = ?");
            $stmt->execute([$enable2FA, $userId]);
            $success = $enable2FA ? 'Two-factor authentication enabled.' : 'Two-factor authentication disabled.';
        }
        
        // Refresh user data
        $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
    }
}

include '../templates/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <?php include 'sidebar.php'; ?>

        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2"><i class="fas fa-cog"></i> Settings</h1>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="fas fa-exclamation-circle"></i> <?php echo e($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <i class="fas fa-check-circle"></i> <?php echo e($success); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <div class="row">
                <div class="col-lg-8">
                    <!-- Account Information -->
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fas fa-user"></i> Account Information</h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="">
                                <?php echo csrfField(); ?>
                                <input type="hidden" name="action" value="update_email">
                                
                                <div class="mb-3">
                                    <label for="email" class="form-label">Email Address</label>
                                    <input type="email" class="form-control" id="email" name="email" 
                                           value="<?php echo e($user['email']); ?>" required>
                                    <?php if (!$user['email_verified']): ?>
                                        <small class="text-warning">
                                            <i class="fas fa-exclamation-triangle"></i> Email not verified
                                        </small>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Account Status</label>
                                    <div>
                                        <span class="badge bg-<?php echo $user['status'] === 'active' ? 'success' : 'warning'; ?>">
                                            <?php echo ucfirst($user['status']); ?>
                                        </span>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Member Since</label>
                                    <div><?php echo date('F d, Y', strtotime($user['created_at'])); ?></div>
                                </div>
                                
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save"></i> Update Email
                                </button>
                            </form>
                        </div>
                    </div>

                    <!-- Change Password -->
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fas fa-lock"></i> Change Password</h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="" id="passwordForm">
                                <?php echo csrfField(); ?>
                                <input type="hidden" name="action" value="change_password">
                                
                                <div class="mb-3">
                                    <label for="current_password" class="form-label">Current Password</label>
                                    <input type="password" class="form-control" id="current_password" 
                                           name="current_password" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="new_password" class="form-label">New Password</label>
                                    <input type="password" class="form-control" id="new_password" 
                                           name="new_password" required minlength="8">
                                    <small class="form-text text-muted">Minimum 8 characters</small>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="confirm_password" class="form-label">Confirm New Password</label>
                                    <input type="password" class="form-control" id="confirm_password" 
                                           name="confirm_password" required minlength="8">
                                </div>
                                
                                <div id="passwordMatch" class="alert alert-danger d-none">
                                    Passwords do not match!
                                </div>
                                
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-key"></i> Change Password
                                </button>
                            </form>
                        </div>
                    </div>

                    <!-- Security Settings -->
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fas fa-shield-alt"></i> Security Settings</h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="">
                                <?php echo csrfField(); ?>
                                <input type="hidden" name="action" value="toggle_2fa">
                                
                                <div class="form-check form-switch mb-3">
                                    <input class="form-check-input" type="checkbox" id="enable_2fa" 
                                           name="enable_2fa" <?php echo $user['two_factor_enabled'] ? 'checked' : ''; ?>
                                           onchange="this.form.submit()">
                                    <label class="form-check-label" for="enable_2fa">
                                        <strong>Two-Factor Authentication (2FA)</strong><br>
                                        <small class="text-muted">Add an extra layer of security to your account</small>
                                    </label>
                                </div>
                                
                                <?php if ($user['two_factor_enabled']): ?>
                                    <div class="alert alert-success">
                                        <i class="fas fa-check-circle"></i> Two-factor authentication is enabled
                                    </div>
                                <?php endif; ?>
                            </form>
                        </div>
                    </div>

                    <!-- Privacy Settings -->
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fas fa-user-shield"></i> Privacy Settings</h5>
                        </div>
                        <div class="card-body">
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" id="profile_visible" checked>
                                <label class="form-check-label" for="profile_visible">
                                    <strong>Make my profile visible to other researchers</strong><br>
                                    <small class="text-muted">Allow other researchers to find and view your profile</small>
                                </label>
                            </div>
                            
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" id="email_notifications" checked>
                                <label class="form-check-label" for="email_notifications">
                                    <strong>Email notifications</strong><br>
                                    <small class="text-muted">Receive email notifications for important updates</small>
                                </label>
                            </div>
                            
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" id="collaboration_requests" checked>
                                <label class="form-check-label" for="collaboration_requests">
                                    <strong>Allow collaboration requests</strong><br>
                                    <small class="text-muted">Let other researchers invite you to collaborate</small>
                                </label>
                            </div>
                            
                            <button type="button" class="btn btn-primary">
                                <i class="fas fa-save"></i> Save Privacy Settings
                            </button>
                        </div>
                    </div>

                    <!-- Danger Zone -->
                    <div class="card shadow border-danger mb-4">
                        <div class="card-header bg-danger text-white">
                            <h5 class="mb-0"><i class="fas fa-exclamation-triangle"></i> Danger Zone</h5>
                        </div>
                        <div class="card-body">
                            <h6>Delete Account</h6>
                            <p class="text-muted">
                                Once you delete your account, there is no going back. Please be certain.
                            </p>
                            <button type="button" class="btn btn-danger" data-bs-toggle="modal" data-bs-target="#deleteAccountModal">
                                <i class="fas fa-trash"></i> Delete My Account
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Sidebar Info -->
                <div class="col-lg-4">
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h6 class="mb-0"><i class="fas fa-info-circle"></i> Account Info</h6>
                        </div>
                        <div class="card-body">
                            <p class="mb-2">
                                <strong>User ID:</strong> #<?php echo $user['id']; ?>
                            </p>
                            <p class="mb-2">
                                <strong>Role:</strong> 
                                <span class="badge bg-primary">Researcher</span>
                            </p>
                            <p class="mb-2">
                                <strong>Last Login:</strong><br>
                                <?php echo $user['last_login'] ? date('M d, Y g:i A', strtotime($user['last_login'])) : 'Never'; ?>
                            </p>
                            <p class="mb-0">
                                <strong>Account Created:</strong><br>
                                <?php echo date('M d, Y', strtotime($user['created_at'])); ?>
                            </p>
                        </div>
                    </div>

                    <div class="card shadow">
                        <div class="card-header">
                            <h6 class="mb-0"><i class="fas fa-question-circle"></i> Need Help?</h6>
                        </div>
                        <div class="card-body">
                            <p class="small mb-2">
                                If you need assistance with your account settings, please contact our support team.
                            </p>
                            <a href="#" class="btn btn-sm btn-outline-primary w-100">
                                <i class="fas fa-life-ring"></i> Contact Support
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<!-- Delete Account Modal -->
<div class="modal fade" id="deleteAccountModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title">Delete Account</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle"></i>
                    <strong>Warning:</strong> This action cannot be undone!
                </div>
                <p>Are you absolutely sure you want to delete your account? This will:</p>
                <ul>
                    <li>Permanently delete your profile and all associated data</li>
                    <li>Remove you from all studies and collaborations</li>
                    <li>Delete all your messages and notifications</li>
                    <li>Cannot be recovered</li>
                </ul>
                <p>Type <strong>DELETE</strong> to confirm:</p>
                <input type="text" class="form-control" id="deleteConfirm" placeholder="Type DELETE">
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmDelete" disabled>
                    Delete My Account
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// Password match validation
document.getElementById('confirm_password').addEventListener('input', function() {
    const newPassword = document.getElementById('new_password').value;
    const confirmPassword = this.value;
    const matchAlert = document.getElementById('passwordMatch');
    
    if (confirmPassword.length > 0) {
        if (newPassword !== confirmPassword) {
            matchAlert.classList.remove('d-none');
        } else {
            matchAlert.classList.add('d-none');
        }
    } else {
        matchAlert.classList.add('d-none');
    }
});

// Delete confirmation
document.getElementById('deleteConfirm').addEventListener('input', function() {
    const confirmBtn = document.getElementById('confirmDelete');
    if (this.value === 'DELETE') {
        confirmBtn.disabled = false;
    } else {
        confirmBtn.disabled = true;
    }
});

document.getElementById('confirmDelete').addEventListener('click', function() {
    alert('Account deletion functionality will be implemented.');
});
</script>

<?php include '../templates/footer.php'; ?>
