<?php
/**
 * Researcher Profile View
 * OBGYN Research & Collaboration Platform
 */

require_once '../config/config.php';

// Check if user is logged in
requireLogin();

// Check if user is a researcher
if (!isResearcher()) {
    redirect(SITE_URL . '/index.php');
}

$pageTitle = 'My Profile';
$isMobile = preg_match('/Mobile|Android|iPhone|iPad|iPod/', $_SERVER['HTTP_USER_AGENT']);
$userId = getCurrentUserId();

// Get researcher profile with all details
$db = getDB();
$stmt = $db->prepare("
    SELECT rp.*, u.first_name, u.last_name, u.email, u.created_at as member_since,
           i.name as institution_name, i.type as institution_type, i.country as institution_country
    FROM researcher_profiles rp
    JOIN users u ON rp.user_id = u.id
    LEFT JOIN institutions i ON rp.institution_id = i.id
    WHERE rp.user_id = ?
");
$stmt->execute([$userId]);
$profile = $stmt->fetch();

// Get user's studies
$stmt = $db->prepare("
    SELECT s.*, 
           (SELECT COUNT(*) FROM study_collaborators WHERE study_id = s.id AND status = 'active') as collaborators_count
    FROM studies s
    WHERE s.creator_id = ? AND s.status != 'archived'
    ORDER BY s.created_at DESC
    LIMIT 10
");
$stmt->execute([$userId]);
$studies = $stmt->fetchAll();

// Get user's publications
$stmt = $db->prepare("
    SELECT * FROM publications 
    WHERE user_id = ? AND status = 'published'
    ORDER BY publication_date DESC
    LIMIT 10
");
$stmt->execute([$userId]);
$publications = $stmt->fetchAll();

// Get connections count
$stmt = $db->prepare("
    SELECT COUNT(*) FROM connections 
    WHERE (requester_id = ? OR receiver_id = ?) AND status = 'accepted'
");
$stmt->execute([$userId, $userId]);
$connectionsCount = $stmt->fetchColumn();

// Include appropriate header based on device
if ($isMobile) {
    include '../templates/researcher-mobile-header.php';
} else {
    include '../templates/researcher-header.php';
}
?>

<div class="container-fluid p-4">
    <div class="row">
        <!-- Main content - full width since sidebar is offcanvas -->
        <main class="col-12">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">
                    <i class="fas fa-user"></i> My Profile
                </h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <a href="edit-profile.php" class="btn btn-primary">
                        <i class="fas fa-edit"></i> Edit Profile
                    </a>
                </div>
            </div>

            <div class="row">
                <!-- Profile Card -->
                <div class="col-lg-4 mb-4">
                    <div class="card shadow">
                        <div class="card-body text-center">
                            <?php if ($profile['photo']): ?>
                                <img src="<?php echo UPLOADS_URL . '/' . e($profile['photo']); ?>" 
                                     alt="Profile Photo" 
                                     class="rounded-circle mb-3" 
                                     style="width: 150px; height: 150px; object-fit: cover;">
                            <?php else: ?>
                                <div class="rounded-circle bg-primary text-white d-inline-flex align-items-center justify-content-center mb-3" 
                                     style="width: 150px; height: 150px; font-size: 3rem;">
                                    <?php echo strtoupper(substr($profile['first_name'], 0, 1) . substr($profile['last_name'], 0, 1)); ?>
                                </div>
                            <?php endif; ?>
                            
                            <h4 class="mb-1">
                                <?php echo e($profile['first_name'] . ' ' . $profile['last_name']); ?>
                                <?php if ($profile['verified_badge']): ?>
                                    <i class="fas fa-check-circle text-primary" title="Verified Researcher"></i>
                                <?php endif; ?>
                            </h4>
                            
                            <?php if ($profile['title']): ?>
                                <p class="text-muted mb-2"><?php echo e($profile['title']); ?></p>
                            <?php endif; ?>
                            
                            <?php if ($profile['specialization']): ?>
                                <p class="mb-2">
                                    <span class="badge bg-primary"><?php echo e($profile['specialization']); ?></span>
                                </p>
                            <?php endif; ?>
                            
                            <?php if ($profile['institution_name']): ?>
                                <p class="text-muted mb-1">
                                    <i class="fas fa-university"></i> <?php echo e($profile['institution_name']); ?>
                                </p>
                            <?php endif; ?>
                            
                            <?php if ($profile['country']): ?>
                                <p class="text-muted mb-3">
                                    <i class="fas fa-map-marker-alt"></i> 
                                    <?php echo e($profile['city'] ? $profile['city'] . ', ' : ''); ?>
                                    <?php echo e($profile['country']); ?>
                                </p>
                            <?php endif; ?>
                            
                            <hr>
                            
                            <!-- Statistics -->
                            <div class="row text-center">
                                <div class="col-4">
                                    <h5 class="mb-0"><?php echo $profile['studies_count']; ?></h5>
                                    <small class="text-muted">Studies</small>
                                </div>
                                <div class="col-4">
                                    <h5 class="mb-0"><?php echo $profile['publications_count']; ?></h5>
                                    <small class="text-muted">Publications</small>
                                </div>
                                <div class="col-4">
                                    <h5 class="mb-0"><?php echo $connectionsCount; ?></h5>
                                    <small class="text-muted">Connections</small>
                                </div>
                            </div>
                            
                            <hr>
                            
                            <!-- Social Links -->
                            <div class="d-flex justify-content-center gap-2">
                                <?php if ($profile['orcid_id']): ?>
                                    <a href="https://orcid.org/<?php echo e($profile['orcid_id']); ?>" 
                                       target="_blank" 
                                       class="btn btn-sm btn-outline-success" 
                                       title="ORCID">
                                        <i class="fab fa-orcid"></i>
                                    </a>
                                <?php endif; ?>
                                
                                <?php if ($profile['google_scholar']): ?>
                                    <a href="<?php echo e($profile['google_scholar']); ?>" 
                                       target="_blank" 
                                       class="btn btn-sm btn-outline-primary" 
                                       title="Google Scholar">
                                        <i class="fas fa-graduation-cap"></i>
                                    </a>
                                <?php endif; ?>
                                
                                <?php if ($profile['researchgate']): ?>
                                    <a href="<?php echo e($profile['researchgate']); ?>" 
                                       target="_blank" 
                                       class="btn btn-sm btn-outline-info" 
                                       title="ResearchGate">
                                        <i class="fab fa-researchgate"></i>
                                    </a>
                                <?php endif; ?>
                                
                                <?php if ($profile['linkedin']): ?>
                                    <a href="<?php echo e($profile['linkedin']); ?>" 
                                       target="_blank" 
                                       class="btn btn-sm btn-outline-primary" 
                                       title="LinkedIn">
                                        <i class="fab fa-linkedin"></i>
                                    </a>
                                <?php endif; ?>
                            </div>
                            
                            <hr>
                            
                            <!-- Contact Info -->
                            <div class="text-start">
                                <p class="mb-2">
                                    <i class="fas fa-envelope text-muted"></i> 
                                    <a href="mailto:<?php echo e($profile['email']); ?>"><?php echo e($profile['email']); ?></a>
                                </p>
                                
                                <?php if ($profile['phone'] && $profile['phone_visible']): ?>
                                    <p class="mb-2">
                                        <i class="fas fa-phone text-muted"></i> 
                                        <?php echo e($profile['phone']); ?>
                                    </p>
                                <?php endif; ?>
                                
                                <p class="mb-0 text-muted small">
                                    <i class="fas fa-calendar text-muted"></i> 
                                    Member since <?php echo date('F Y', strtotime($profile['member_since'])); ?>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Profile Details -->
                <div class="col-lg-8 mb-4">
                    <!-- Bio -->
                    <?php if ($profile['bio']): ?>
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fas fa-user-circle"></i> About</h5>
                        </div>
                        <div class="card-body">
                            <p class="mb-0"><?php echo nl2br(e($profile['bio'])); ?></p>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Research Interests -->
                    <?php if ($profile['research_interests']): ?>
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fas fa-lightbulb"></i> Research Interests</h5>
                        </div>
                        <div class="card-body">
                            <p class="mb-0"><?php echo nl2br(e($profile['research_interests'])); ?></p>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Degrees -->
                    <?php if ($profile['degrees']): ?>
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fas fa-graduation-cap"></i> Education & Degrees</h5>
                        </div>
                        <div class="card-body">
                            <p class="mb-0"><?php echo nl2br(e($profile['degrees'])); ?></p>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Studies -->
                    <div class="card shadow mb-4">
                        <div class="card-header d-flex justify-content-between align-items-center">
                            <h5 class="mb-0"><i class="fas fa-flask"></i> Research Studies</h5>
                            <a href="studies.php" class="btn btn-sm btn-outline-primary">View All</a>
                        </div>
                        <div class="card-body">
                            <?php if (empty($studies)): ?>
                                <p class="text-muted text-center py-3">No studies yet.</p>
                            <?php else: ?>
                                <div class="list-group list-group-flush">
                                    <?php foreach ($studies as $study): ?>
                                    <div class="list-group-item">
                                        <div class="d-flex w-100 justify-content-between">
                                            <h6 class="mb-1">
                                                <a href="edit-study.php?id=<?php echo $study['id']; ?>">
                                                    <?php echo e($study['title']); ?>
                                                </a>
                                            </h6>
                                            <small class="text-muted"><?php echo date('M Y', strtotime($study['created_at'])); ?></small>
                                        </div>
                                        <p class="mb-1 small text-muted">
                                            <?php echo e(substr($study['summary'] ?? 'No summary available', 0, 150)); ?>...
                                        </p>
                                        <div class="d-flex gap-2">
                                            <span class="badge bg-secondary"><?php echo ucfirst($study['type']); ?></span>
                                            <span class="badge bg-info"><?php echo ucfirst($study['status']); ?></span>
                                            <?php if ($study['collaborators_count'] > 0): ?>
                                                <span class="badge bg-success">
                                                    <i class="fas fa-users"></i> <?php echo $study['collaborators_count']; ?> Collaborators
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Publications -->
                    <div class="card shadow mb-4">
                        <div class="card-header">
                            <h5 class="mb-0"><i class="fas fa-book"></i> Publications</h5>
                        </div>
                        <div class="card-body">
                            <?php if (empty($publications)): ?>
                                <p class="text-muted text-center py-3">No publications yet.</p>
                            <?php else: ?>
                                <div class="list-group list-group-flush">
                                    <?php foreach ($publications as $pub): ?>
                                    <div class="list-group-item">
                                        <h6 class="mb-1"><?php echo e($pub['title']); ?></h6>
                                        <p class="mb-1 small text-muted"><?php echo e($pub['authors']); ?></p>
                                        <p class="mb-1 small">
                                            <?php if ($pub['journal']): ?>
                                                <em><?php echo e($pub['journal']); ?></em>
                                            <?php endif; ?>
                                            <?php if ($pub['publication_date']): ?>
                                                (<?php echo date('Y', strtotime($pub['publication_date'])); ?>)
                                            <?php endif; ?>
                                        </p>
                                        <?php if ($pub['doi']): ?>
                                            <p class="mb-0 small">
                                                <strong>DOI:</strong> 
                                                <a href="https://doi.org/<?php echo e($pub['doi']); ?>" target="_blank">
                                                    <?php echo e($pub['doi']); ?>
                                                </a>
                                            </p>
                                        <?php endif; ?>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>



<?php // Include appropriate footer based on device
if ($isMobile) {
    include '../templates/researcher-mobile-footer.php';
} else {
    include '../templates/footer.php';
} ?>
