<?php
/**
 * Study Detail - Public Page
 * OBGYN Research & Collaboration Platform
 */

require_once '../config/config.php';

$isMobile = preg_match('/Mobile|Android|iPhone|iPad|iPod/', $_SERVER['HTTP_USER_AGENT']);

$studyId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$studyId) {
    redirect(SITE_URL . '/public/studies.php');
}

$db = getDB();

// Get study details
$stmt = $db->prepare("
    SELECT s.*, u.first_name, u.last_name, u.id as creator_user_id,
           c.name as category_name, i.name as institution_name
    FROM studies s
    JOIN users u ON s.creator_id = u.id
    LEFT JOIN categories c ON s.category_id = c.id
    LEFT JOIN institutions i ON s.institution_id = i.id
    WHERE s.id = ? AND s.is_public = 1 AND s.status IN ('active', 'completed') AND s.approval_status = 'approved'
");
$stmt->execute([$studyId]);
$study = $stmt->fetch();

if (!$study) {
    redirect(SITE_URL . '/public/studies.php');
}

$pageTitle = $study['title'];

// Get collaborators
$stmt = $db->prepare("
    SELECT sc.*, u.first_name, u.last_name, u.id as user_id
    FROM study_collaborators sc
    JOIN users u ON sc.user_id = u.id
    WHERE sc.study_id = ? AND sc.status = 'active'
    ORDER BY sc.role, u.first_name
");
$stmt->execute([$studyId]);
$collaborators = $stmt->fetchAll();

// Get related studies
$stmt = $db->prepare("
    SELECT s.*, u.first_name, u.last_name
    FROM studies s
    JOIN users u ON s.creator_id = u.id
    WHERE s.id != ? AND s.type = ? AND s.is_public = 1 AND s.status = 'active'
    ORDER BY s.created_at DESC
    LIMIT 3
");
$stmt->execute([$studyId, $study['type']]);
$relatedStudies = $stmt->fetchAll();

// Include appropriate header based on device
if ($isMobile) {
    include '../templates/mobile-header.php';
} else {
    include '../templates/header.php';
}
?>

<div class="container my-5">
    <nav aria-label="breadcrumb">
        <ol class="breadcrumb">
            <li class="breadcrumb-item"><a href="<?php echo SITE_URL; ?>">Home</a></li>
            <li class="breadcrumb-item"><a href="studies.php">Studies</a></li>
            <li class="breadcrumb-item active"><?php echo e(substr($study['title'], 0, 50)); ?></li>
        </ol>
    </nav>
    
    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow-sm mb-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start mb-3">
                        <div>
                            <span class="badge bg-<?php echo $study['type'] == 'clinical' ? 'primary' : 'success'; ?> me-2">
                                <?php echo ucfirst($study['type']); ?>
                            </span>
                            <?php if ($study['category_name']): ?>
                                <span class="badge bg-secondary"><?php echo e($study['category_name']); ?></span>
                            <?php endif; ?>
                        </div>
                        <span class="badge bg-<?php echo $study['status'] == 'active' ? 'success' : 'info'; ?>">
                            <?php echo ucfirst($study['status']); ?>
                        </span>
                    </div>
                    
                    <h1 class="h2 mb-3"><?php echo e($study['title']); ?></h1>
                    
                    <div class="study-meta mb-4">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <small class="text-muted d-block">
                                    <i class="fas fa-user"></i> <strong>Principal Investigator:</strong><br>
                                    <a href="researcher-profile.php?id=<?php echo $study['creator_user_id']; ?>" class="text-decoration-none">
                                        <?php echo e($study['first_name'] . ' ' . $study['last_name']); ?>
                                    </a>
                                </small>
                            </div>
                            <?php if ($study['institution_name']): ?>
                            <div class="col-md-6">
                                <small class="text-muted d-block">
                                    <i class="fas fa-university"></i> <strong>Institution:</strong><br>
                                    <?php echo e($study['institution_name']); ?>
                                </small>
                            </div>
                            <?php endif; ?>
                            <div class="col-md-6">
                                <small class="text-muted d-block">
                                    <i class="fas fa-calendar"></i> <strong>Start Date:</strong><br>
                                    <?php echo date('F d, Y', strtotime($study['start_date'])); ?>
                                </small>
                            </div>
                            <?php if ($study['end_date']): ?>
                            <div class="col-md-6">
                                <small class="text-muted d-block">
                                    <i class="fas fa-calendar-check"></i> <strong>End Date:</strong><br>
                                    <?php echo date('F d, Y', strtotime($study['end_date'])); ?>
                                </small>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <hr>
                    
                    <h4 class="mb-3">Summary</h4>
                    <p class="lead"><?php echo nl2br(e($study['summary'])); ?></p>
                    
                    <?php if (isset($study['description']) && $study['description']): ?>
                    <hr>
                    <h4 class="mb-3">Description</h4>
                    <div class="study-description">
                        <?php echo nl2br(e($study['description'])); ?>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($study['objectives']): ?>
                    <hr>
                    <h4 class="mb-3">Objectives</h4>
                    <div class="study-objectives">
                        <?php echo nl2br(e($study['objectives'])); ?>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($study['methodology']): ?>
                    <hr>
                    <h4 class="mb-3">Methodology</h4>
                    <div class="study-methodology">
                        <?php echo nl2br(e($study['methodology'])); ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Collaborators -->
            <?php if (!empty($collaborators)): ?>
            <div class="card shadow-sm mb-4">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-users"></i> Collaborators (<?php echo count($collaborators); ?>)</h5>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <?php foreach ($collaborators as $collab): ?>
                        <div class="col-md-6">
                            <div class="d-flex align-items-center">
                                <div class="avatar-circle me-3">
                                    <?php echo strtoupper(substr($collab['first_name'], 0, 1) . substr($collab['last_name'], 0, 1)); ?>
                                </div>
                                <div>
                                    <a href="researcher-profile.php?id=<?php echo $collab['user_id']; ?>" class="text-decoration-none">
                                        <strong><?php echo e($collab['first_name'] . ' ' . $collab['last_name']); ?></strong>
                                    </a>
                                    <br>
                                    <small class="text-muted"><?php echo ucfirst(str_replace('_', ' ', $collab['role'])); ?></small>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
        
        <div class="col-lg-4">
            <!-- Contact Card -->
            <?php if (isLoggedIn()): ?>
            <div class="card shadow-sm mb-4">
                <div class="card-body text-center">
                    <h5 class="card-title">Interested in this study?</h5>
                    <p class="card-text small text-muted">Connect with the research team</p>
                    <a href="<?php echo SITE_URL; ?>/researcher/messages.php?to=<?php echo $study['creator_user_id']; ?>" class="btn btn-primary w-100">
                        <i class="fas fa-envelope"></i> Contact Researcher
                    </a>
                </div>
            </div>
            <?php else: ?>
            <div class="card shadow-sm mb-4">
                <div class="card-body text-center">
                    <h5 class="card-title">Interested in this study?</h5>
                    <p class="card-text small text-muted">Login to connect with researchers</p>
                    <a href="<?php echo SITE_URL; ?>/login.php" class="btn btn-primary w-100">
                        <i class="fas fa-sign-in-alt"></i> Login
                    </a>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Study Info -->
            <div class="card shadow-sm mb-4">
                <div class="card-header">
                    <h6 class="mb-0">Study Information</h6>
                </div>
                <div class="card-body">
                    <ul class="list-unstyled mb-0">
                        <li class="mb-2">
                            <i class="fas fa-tag text-muted"></i>
                            <strong>Type:</strong> <?php echo ucfirst($study['type']); ?>
                        </li>
                        <li class="mb-2">
                            <i class="fas fa-signal text-muted"></i>
                            <strong>Status:</strong> <?php echo ucfirst($study['status']); ?>
                        </li>
                        <li class="mb-2">
                            <i class="fas fa-users text-muted"></i>
                            <strong>Collaborators:</strong> <?php echo count($collaborators); ?>
                        </li>
                        <li class="mb-2">
                            <i class="fas fa-clock text-muted"></i>
                            <strong>Created:</strong> <?php echo date('M d, Y', strtotime($study['created_at'])); ?>
                        </li>
                        <li>
                            <i class="fas fa-sync text-muted"></i>
                            <strong>Updated:</strong> <?php echo timeAgo($study['updated_at']); ?>
                        </li>
                    </ul>
                </div>
            </div>
            
            <!-- Related Studies -->
            <?php if (!empty($relatedStudies)): ?>
            <div class="card shadow-sm">
                <div class="card-header">
                    <h6 class="mb-0">Related Studies</h6>
                </div>
                <div class="list-group list-group-flush">
                    <?php foreach ($relatedStudies as $related): ?>
                    <a href="study-detail.php?id=<?php echo $related['id']; ?>" class="list-group-item list-group-item-action">
                        <h6 class="mb-1"><?php echo e(substr($related['title'], 0, 60)); ?><?php echo strlen($related['title']) > 60 ? '...' : ''; ?></h6>
                        <small class="text-muted">
                            By <?php echo e($related['first_name'] . ' ' . $related['last_name']); ?>
                        </small>
                    </a>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
.avatar-circle {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 18px;
}
</style>

<?php // Include appropriate footer based on device
if ($isMobile) {
    include '../templates/mobile-footer.php';
} else {
    include '../templates/footer.php';
} ?>
