<?php
/**
 * Create Dummy Blog Posts
 * Run this file once to populate the database with sample blog content
 */


require_once '../config/config.php';
require_once '../includes/rbac.php';

// Check if user is admin
requireLogin();
if (!isAdmin()) {
    die('Access denied. Admin only.');
}

$db = getDB();
$success = [];
$errors = [];

echo "<!DOCTYPE html>
<html>
<head>
    <title>Create Blog Dummy Data</title>
    <link href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css' rel='stylesheet'>
</head>
<body>
<div class='container mt-5'>
    <h1>Creating Blog Dummy Data</h1>
    <div class='progress mb-3'>
        <div class='progress-bar progress-bar-striped progress-bar-animated' role='progressbar' style='width: 0%' id='progress'></div>
    </div>
    <div id='output'>";

try {
    // Get current user ID for author
    $authorId = getCurrentUserId();
    
    // Blog posts data
    $blogPosts = [
        [
            'title' => 'Breakthrough in Prenatal Care: New Guidelines Released',
            'slug' => 'breakthrough-prenatal-care-guidelines',
            'excerpt' => 'The American College of Obstetricians and Gynecologists has released updated guidelines for prenatal care that promise to improve maternal and fetal outcomes.',
            'content' => '<h2>Revolutionary Changes in Prenatal Care</h2>
<p>The American College of Obstetricians and Gynecologists (ACOG) has released comprehensive new guidelines for prenatal care that represent the most significant update in over a decade. These guidelines incorporate the latest research findings and aim to improve outcomes for both mothers and babies.</p>

<h3>Key Updates Include:</h3>
<ul>
<li><strong>Enhanced Screening Protocols:</strong> New recommendations for genetic screening and early detection of potential complications</li>
<li><strong>Mental Health Integration:</strong> Mandatory screening for perinatal depression and anxiety at multiple points during pregnancy</li>
<li><strong>Nutrition Guidelines:</strong> Updated dietary recommendations based on recent nutritional research</li>
<li><strong>Exercise Recommendations:</strong> Evidence-based guidelines for physical activity during pregnancy</li>
</ul>

<h3>Impact on Clinical Practice</h3>
<p>Healthcare providers are encouraged to implement these guidelines immediately. The changes are expected to reduce maternal mortality rates and improve overall pregnancy outcomes. Training programs and resources are being made available to support the transition.</p>

<h3>Patient Education</h3>
<p>Expectant mothers should discuss these new guidelines with their healthcare providers. The updated protocols may mean additional screenings or changes to their prenatal care routine, all designed to ensure the healthiest possible pregnancy.</p>

<blockquote class="blockquote">
<p>"These guidelines represent a paradigm shift in how we approach prenatal care, putting evidence-based medicine at the forefront of maternal health."</p>
<footer class="blockquote-footer">Dr. Sarah Johnson, ACOG President</footer>
</blockquote>',
            'featured_image' => 'https://images.unsplash.com/photo-1584515933487-779824d29309?w=1200',
            'meta_title' => 'New Prenatal Care Guidelines 2025 | OBGYN Research',
            'meta_description' => 'Discover the latest prenatal care guidelines from ACOG that are revolutionizing maternal healthcare.',
            'meta_keywords' => 'prenatal care, pregnancy guidelines, ACOG, maternal health',
            'featured' => 1,
            'views' => rand(500, 2000)
        ],
        [
            'title' => 'Advances in Minimally Invasive Gynecologic Surgery',
            'slug' => 'advances-minimally-invasive-surgery',
            'excerpt' => 'Recent technological advances are making gynecologic surgeries safer, less invasive, and more effective than ever before.',
            'content' => '<h2>The Future of Gynecologic Surgery</h2>
<p>Minimally invasive surgical techniques have transformed gynecologic care over the past decade. New technologies and refined procedures are offering patients faster recovery times, reduced pain, and better outcomes.</p>

<h3>Latest Innovations</h3>
<ul>
<li><strong>Robotic-Assisted Surgery:</strong> Enhanced precision and control for complex procedures</li>
<li><strong>Single-Port Laparoscopy:</strong> Reduced scarring and faster recovery</li>
<li><strong>Advanced Imaging:</strong> Real-time 3D visualization during procedures</li>
<li><strong>Energy Devices:</strong> Improved hemostasis and tissue handling</li>
</ul>

<h3>Patient Benefits</h3>
<p>Patients undergoing minimally invasive procedures typically experience:</p>
<ul>
<li>Shorter hospital stays (often same-day discharge)</li>
<li>Less postoperative pain</li>
<li>Faster return to normal activities</li>
<li>Reduced risk of complications</li>
<li>Minimal scarring</li>
</ul>

<h3>Training and Adoption</h3>
<p>Medical institutions worldwide are investing in training programs to ensure surgeons are proficient in these advanced techniques. Simulation-based training and mentorship programs are helping to accelerate adoption while maintaining high safety standards.</p>',
            'featured_image' => 'https://images.unsplash.com/photo-1579154204601-01588f351e67?w=1200',
            'meta_title' => 'Minimally Invasive Gynecologic Surgery Advances',
            'meta_description' => 'Explore the latest advances in minimally invasive gynecologic surgery and their benefits for patients.',
            'meta_keywords' => 'minimally invasive surgery, gynecologic surgery, robotic surgery, laparoscopy',
            'featured' => 1,
            'views' => rand(300, 1500)
        ],
        [
            'title' => 'Understanding Endometriosis: Latest Research Findings',
            'slug' => 'understanding-endometriosis-research',
            'excerpt' => 'New research is shedding light on the causes, diagnosis, and treatment of endometriosis, offering hope to millions of women worldwide.',
            'content' => '<h2>Endometriosis: A Complex Condition</h2>
<p>Endometriosis affects approximately 10% of women of reproductive age, yet it remains one of the most challenging gynecologic conditions to diagnose and treat. Recent research is providing new insights into this complex disease.</p>

<h3>Recent Research Breakthroughs</h3>
<ul>
<li><strong>Genetic Factors:</strong> Identification of genetic markers associated with increased risk</li>
<li><strong>Inflammatory Pathways:</strong> Understanding the role of inflammation in disease progression</li>
<li><strong>Diagnostic Biomarkers:</strong> Development of non-invasive diagnostic tests</li>
<li><strong>Novel Treatments:</strong> New therapeutic approaches targeting specific disease mechanisms</li>
</ul>

<h3>Improved Diagnostic Approaches</h3>
<p>Traditional diagnosis of endometriosis required surgical intervention. New research is focusing on developing non-invasive diagnostic methods, including:</p>
<ul>
<li>Blood-based biomarker tests</li>
<li>Advanced imaging techniques</li>
<li>Saliva and urine tests</li>
</ul>

<h3>Treatment Options</h3>
<p>Current treatment strategies include:</p>
<ul>
<li>Hormonal therapies</li>
<li>Pain management</li>
<li>Surgical intervention</li>
<li>Lifestyle modifications</li>
<li>Complementary therapies</li>
</ul>

<p>Ongoing clinical trials are investigating new medications and treatment approaches that may offer better outcomes with fewer side effects.</p>',
            'featured_image' => 'https://images.unsplash.com/photo-1576091160399-112ba8d25d1d?w=1200',
            'meta_title' => 'Endometriosis Research: Latest Findings and Treatments',
            'meta_description' => 'Learn about the latest research findings on endometriosis, including new diagnostic methods and treatment options.',
            'meta_keywords' => 'endometriosis, gynecologic research, women\'s health, chronic pain',
            'featured' => 0,
            'views' => rand(400, 1800)
        ],
        [
            'title' => 'Maternal Mental Health: Breaking the Stigma',
            'slug' => 'maternal-mental-health-stigma',
            'excerpt' => 'Addressing maternal mental health is crucial for the well-being of both mothers and babies. Learn about the latest initiatives and support systems.',
            'content' => '<h2>The Importance of Maternal Mental Health</h2>
<p>Perinatal mental health conditions affect up to 20% of women during pregnancy and the first year postpartum. Despite their prevalence, these conditions often go undiagnosed and untreated due to stigma and lack of awareness.</p>

<h3>Common Perinatal Mental Health Conditions</h3>
<ul>
<li><strong>Perinatal Depression:</strong> Affecting 10-15% of pregnant and postpartum women</li>
<li><strong>Perinatal Anxiety:</strong> Often co-occurring with depression</li>
<li><strong>Postpartum Psychosis:</strong> A rare but serious condition requiring immediate treatment</li>
<li><strong>Birth Trauma and PTSD:</strong> Following difficult or traumatic births</li>
</ul>

<h3>Breaking Down Barriers</h3>
<p>Healthcare systems are implementing new approaches to address maternal mental health:</p>
<ul>
<li>Universal screening at prenatal and postpartum visits</li>
<li>Integrated care models combining obstetric and mental health services</li>
<li>Peer support programs</li>
<li>Telehealth options for increased accessibility</li>
<li>Partner and family education</li>
</ul>

<h3>Treatment Options</h3>
<p>Evidence-based treatments include:</p>
<ul>
<li>Psychotherapy (CBT, IPT)</li>
<li>Medication when appropriate</li>
<li>Support groups</li>
<li>Lifestyle interventions</li>
<li>Complementary therapies</li>
</ul>

<h3>Resources and Support</h3>
<p>If you or someone you know is struggling with perinatal mental health issues, help is available. Contact your healthcare provider or reach out to organizations like Postpartum Support International for resources and support.</p>',
            'featured_image' => 'https://images.unsplash.com/photo-1559757175-5700dde675bc?w=1200',
            'meta_title' => 'Maternal Mental Health: Support and Resources',
            'meta_description' => 'Understanding maternal mental health, breaking stigma, and finding support for perinatal depression and anxiety.',
            'meta_keywords' => 'maternal mental health, postpartum depression, perinatal anxiety, maternal wellness',
            'featured' => 1,
            'views' => rand(600, 2500)
        ],
        [
            'title' => 'Fertility Preservation: Options and Advances',
            'slug' => 'fertility-preservation-options',
            'excerpt' => 'Modern fertility preservation techniques are giving women more control over their reproductive futures. Explore the latest options and success rates.',
            'content' => '<h2>Fertility Preservation: Empowering Reproductive Choices</h2>
<p>Advances in reproductive medicine have made fertility preservation more accessible and successful than ever before. Whether for medical reasons or personal choice, women now have multiple options to preserve their fertility.</p>

<h3>Fertility Preservation Methods</h3>
<ul>
<li><strong>Egg Freezing (Oocyte Cryopreservation):</strong> The most common method for women</li>
<li><strong>Embryo Freezing:</strong> For women with partners or using donor sperm</li>
<li><strong>Ovarian Tissue Freezing:</strong> Experimental option for prepubertal girls or urgent cases</li>
<li><strong>Ovarian Suppression:</strong> During cancer treatment</li>
</ul>

<h3>Who Should Consider Fertility Preservation?</h3>
<ul>
<li>Women facing cancer treatment</li>
<li>Those with endometriosis or other conditions affecting fertility</li>
<li>Women planning to delay childbearing</li>
<li>Transgender individuals before gender-affirming treatment</li>
<li>Those with family history of early menopause</li>
</ul>

<h3>Success Rates and Considerations</h3>
<p>Success rates for fertility preservation have improved dramatically:</p>
<ul>
<li>Egg freezing success rates now comparable to fresh eggs</li>
<li>Vitrification technique has improved survival rates to over 90%</li>
<li>Age at freezing is the most important factor for success</li>
</ul>

<h3>The Process</h3>
<p>Typical fertility preservation involves:</p>
<ol>
<li>Initial consultation and testing</li>
<li>Ovarian stimulation (10-14 days)</li>
<li>Egg retrieval procedure</li>
<li>Freezing and storage</li>
</ol>

<h3>Cost and Insurance</h3>
<p>While costs vary, many insurance plans now cover fertility preservation for medical reasons. Some employers are also adding elective fertility preservation to their benefits packages.</p>',
            'featured_image' => 'https://images.unsplash.com/photo-1582719471384-894fbb16e074?w=1200',
            'meta_title' => 'Fertility Preservation: Options, Success Rates, and Costs',
            'meta_description' => 'Comprehensive guide to fertility preservation options including egg freezing, success rates, and what to expect.',
            'meta_keywords' => 'fertility preservation, egg freezing, reproductive health, family planning',
            'featured' => 0,
            'views' => rand(350, 1600)
        ],
        [
            'title' => 'HPV Vaccination: Protecting Against Cervical Cancer',
            'slug' => 'hpv-vaccination-cervical-cancer',
            'excerpt' => 'HPV vaccination has dramatically reduced cervical cancer rates. Learn about the latest recommendations and the impact of vaccination programs.',
            'content' => '<h2>HPV Vaccination: A Cancer Prevention Success Story</h2>
<p>The human papillomavirus (HPV) vaccine represents one of the most significant advances in cancer prevention. Since its introduction, cervical cancer rates have dropped dramatically in vaccinated populations.</p>

<h3>Understanding HPV and Cervical Cancer</h3>
<p>HPV is the most common sexually transmitted infection, and certain high-risk types can lead to cervical cancer. The vaccine protects against the types responsible for about 90% of cervical cancers.</p>

<h3>Current Vaccination Recommendations</h3>
<ul>
<li><strong>Routine vaccination:</strong> Ages 11-12 (can start as early as age 9)</li>
<li><strong>Catch-up vaccination:</strong> Through age 26 for those not previously vaccinated</li>
<li><strong>Shared decision-making:</strong> Ages 27-45 for some adults</li>
</ul>

<h3>Vaccine Effectiveness</h3>
<p>Research shows impressive results:</p>
<ul>
<li>Nearly 100% effective in preventing HPV types covered by the vaccine</li>
<li>90% reduction in cervical precancers in vaccinated populations</li>
<li>Herd immunity benefits for unvaccinated individuals</li>
<li>Protection appears to be long-lasting</li>
</ul>

<h3>Safety Profile</h3>
<p>The HPV vaccine has an excellent safety record:</p>
<ul>
<li>Over 270 million doses administered worldwide</li>
<li>Extensive safety monitoring</li>
<li>Most common side effects are mild (arm soreness, headache)</li>
<li>No evidence of serious long-term side effects</li>
</ul>

<h3>Global Impact</h3>
<p>Countries with high vaccination rates are seeing dramatic reductions in cervical cancer. Australia is on track to eliminate cervical cancer as a public health problem by 2035.</p>

<h3>Barriers to Vaccination</h3>
<p>Despite its effectiveness, vaccination rates remain suboptimal in many areas due to:</p>
<ul>
<li>Lack of awareness</li>
<li>Access issues</li>
<li>Misinformation</li>
<li>Healthcare provider recommendations</li>
</ul>

<p>Efforts are underway to improve access and education to ensure all eligible individuals can benefit from this life-saving vaccine.</p>',
            'featured_image' => 'https://images.unsplash.com/photo-1631217868264-e5b90bb7e133?w=1200',
            'meta_title' => 'HPV Vaccination: Preventing Cervical Cancer',
            'meta_description' => 'Learn about HPV vaccination, its effectiveness in preventing cervical cancer, and current recommendations.',
            'meta_keywords' => 'HPV vaccine, cervical cancer prevention, vaccination, women\'s health',
            'featured' => 0,
            'views' => rand(450, 1900)
        ],
        [
            'title' => 'Polycystic Ovary Syndrome: Management Strategies',
            'slug' => 'pcos-management-strategies',
            'excerpt' => 'PCOS affects millions of women worldwide. Discover evidence-based management strategies and lifestyle interventions that can make a difference.',
            'content' => '<h2>Managing PCOS: A Comprehensive Approach</h2>
<p>Polycystic Ovary Syndrome (PCOS) is one of the most common endocrine disorders affecting women of reproductive age. While there is no cure, effective management strategies can significantly improve symptoms and long-term health outcomes.</p>

<h3>Understanding PCOS</h3>
<p>PCOS is characterized by:</p>
<ul>
<li>Irregular menstrual cycles</li>
<li>Elevated androgen levels</li>
<li>Polycystic ovaries on ultrasound</li>
<li>Insulin resistance</li>
<li>Increased risk of metabolic syndrome</li>
</ul>

<h3>Lifestyle Interventions</h3>
<p>Research shows that lifestyle modifications can be highly effective:</p>

<h4>Diet</h4>
<ul>
<li>Low glycemic index foods</li>
<li>Balanced macronutrients</li>
<li>Anti-inflammatory foods</li>
<li>Adequate protein intake</li>
</ul>

<h4>Exercise</h4>
<ul>
<li>Regular physical activity (150 minutes/week)</li>
<li>Combination of cardio and strength training</li>
<li>High-intensity interval training (HIIT)</li>
</ul>

<h4>Weight Management</h4>
<ul>
<li>Even 5-10% weight loss can improve symptoms</li>
<li>Focus on sustainable changes</li>
<li>Address emotional eating</li>
</ul>

<h3>Medical Management</h3>
<p>Treatment options include:</p>

<h4>For Irregular Periods</h4>
<ul>
<li>Combined oral contraceptives</li>
<li>Progestin therapy</li>
</ul>

<h4>For Insulin Resistance</h4>
<ul>
<li>Metformin</li>
<li>Inositol supplements</li>
</ul>

<h4>For Excess Hair Growth</h4>
<ul>
<li>Anti-androgen medications</li>
<li>Topical treatments</li>
<li>Hair removal procedures</li>
</ul>

<h4>For Fertility</h4>
<ul>
<li>Ovulation induction medications</li>
<li>Assisted reproductive technologies</li>
</ul>

<h3>Long-term Health Considerations</h3>
<p>Women with PCOS have increased risk for:</p>
<ul>
<li>Type 2 diabetes</li>
<li>Cardiovascular disease</li>
<li>Endometrial cancer</li>
<li>Sleep apnea</li>
</ul>

<p>Regular screening and preventive care are essential for long-term health.</p>

<h3>Mental Health Support</h3>
<p>PCOS can impact mental health. Support options include:</p>
<ul>
<li>Counseling and therapy</li>
<li>Support groups</li>
<li>Stress management techniques</li>
<li>Treatment for anxiety/depression if needed</li>
</ul>',
            'featured_image' => 'https://images.unsplash.com/photo-1576091160550-2173dba999ef?w=1200',
            'meta_title' => 'PCOS Management: Lifestyle and Medical Strategies',
            'meta_description' => 'Comprehensive guide to managing PCOS through lifestyle interventions and medical treatments.',
            'meta_keywords' => 'PCOS, polycystic ovary syndrome, hormone health, fertility, insulin resistance',
            'featured' => 0,
            'views' => rand(500, 2200)
        ],
        [
            'title' => 'Menopause Management: Beyond Hormone Therapy',
            'slug' => 'menopause-management-options',
            'excerpt' => 'Explore the full spectrum of menopause management options, from hormone therapy to lifestyle interventions and alternative treatments.',
            'content' => '<h2>Navigating Menopause: A Holistic Approach</h2>
<p>Menopause is a natural transition that affects every woman differently. Modern medicine offers a range of options to manage symptoms and maintain quality of life during this important life stage.</p>

<h3>Understanding Menopause</h3>
<p>Menopause typically occurs between ages 45-55 and involves:</p>
<ul>
<li>Declining estrogen and progesterone levels</li>
<li>Cessation of menstrual periods</li>
<li>Various physical and emotional changes</li>
</ul>

<h3>Common Symptoms</h3>
<ul>
<li>Hot flashes and night sweats</li>
<li>Sleep disturbances</li>
<li>Mood changes</li>
<li>Vaginal dryness</li>
<li>Decreased libido</li>
<li>Weight gain</li>
<li>Cognitive changes</li>
</ul>

<h3>Hormone Therapy (HT)</h3>
<p>For many women, hormone therapy remains the most effective treatment for moderate to severe symptoms:</p>

<h4>Types of HT</h4>
<ul>
<li>Estrogen-only (for women without a uterus)</li>
<li>Combined estrogen-progestogen (for women with a uterus)</li>
<li>Various delivery methods (pills, patches, gels, rings)</li>
</ul>

<h4>Benefits</h4>
<ul>
<li>Effective relief of hot flashes</li>
<li>Prevention of bone loss</li>
<li>Improvement in vaginal symptoms</li>
<li>Potential cardiovascular benefits when started early</li>
</ul>

<h4>Considerations</h4>
<ul>
<li>Individual risk-benefit assessment</li>
<li>Timing of initiation matters</li>
<li>Regular reassessment</li>
<li>Lowest effective dose</li>
</ul>

<h3>Non-Hormonal Options</h3>

<h4>Medications</h4>
<ul>
<li>SSRIs/SNRIs for hot flashes and mood</li>
<li>Gabapentin for hot flashes</li>
<li>Vaginal moisturizers and lubricants</li>
<li>Ospemifene for vaginal symptoms</li>
</ul>

<h4>Lifestyle Interventions</h4>
<ul>
<li>Regular exercise</li>
<li>Healthy diet</li>
<li>Weight management</li>
<li>Stress reduction</li>
<li>Adequate sleep</li>
<li>Smoking cessation</li>
<li>Limited alcohol</li>
</ul>

<h4>Complementary Therapies</h4>
<ul>
<li>Cognitive behavioral therapy</li>
<li>Mindfulness and meditation</li>
<li>Acupuncture</li>
<li>Yoga</li>
</ul>

<h3>Bone Health</h3>
<p>Menopause increases osteoporosis risk. Prevention strategies include:</p>
<ul>
<li>Calcium and vitamin D supplementation</li>
<li>Weight-bearing exercise</li>
<li>Bone density screening</li>
<li>Medications if needed</li>
</ul>

<h3>Cardiovascular Health</h3>
<p>Heart disease risk increases after menopause. Focus on:</p>
<ul>
<li>Blood pressure control</li>
<li>Cholesterol management</li>
<li>Diabetes prevention/management</li>
<li>Heart-healthy lifestyle</li>
</ul>

<h3>Sexual Health</h3>
<p>Maintaining sexual health during menopause:</p>
<ul>
<li>Open communication with partners</li>
<li>Vaginal estrogen therapy</li>
<li>Lubricants and moisturizers</li>
<li>Pelvic floor therapy</li>
<li>Addressing psychological factors</li>
</ul>',
            'featured_image' => 'https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=1200',
            'meta_title' => 'Menopause Management: Comprehensive Treatment Options',
            'meta_description' => 'Complete guide to menopause management including hormone therapy, lifestyle changes, and alternative treatments.',
            'meta_keywords' => 'menopause, hormone therapy, hot flashes, women\'s health, midlife health',
            'featured' => 0,
            'views' => rand(400, 1700)
        ]
    ];
    
    echo "<p class='text-info'>Starting to create blog posts...</p>";
    
    $total = count($blogPosts);
    $current = 0;
    
    foreach ($blogPosts as $post) {
        $current++;
        $progress = ($current / $total) * 100;
        
        echo "<script>document.getElementById('progress').style.width = '{$progress}%';</script>";
        echo "<p>Creating post {$current}/{$total}: <strong>{$post['title']}</strong>...</p>";
        flush();
        
        try {
            // Check if post already exists
            $stmt = $db->prepare("SELECT id FROM cms_posts WHERE slug = ?");
            $stmt->execute([$post['slug']]);
            
            if ($stmt->fetch()) {
                echo "<p class='text-warning'>⚠️ Post already exists: {$post['title']}</p>";
                continue;
            }
            
            // Insert blog post
            $stmt = $db->prepare("
                INSERT INTO cms_posts (
                    title, slug, content, excerpt, featured_image,
                    author_id, status, featured, allow_comments,
                    meta_title, meta_description, meta_keywords,
                    views, published_at, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, 'published', ?, 1, ?, ?, ?, ?, NOW(), NOW())
            ");
            
            $stmt->execute([
                $post['title'],
                $post['slug'],
                $post['content'],
                $post['excerpt'],
                $post['featured_image'],
                $authorId,
                $post['featured'],
                $post['meta_title'],
                $post['meta_description'],
                $post['meta_keywords'],
                $post['views']
            ]);
            
            echo "<p class='text-success'>✓ Created: {$post['title']}</p>";
            $success[] = $post['title'];
            
        } catch (Exception $e) {
            echo "<p class='text-danger'>✗ Error creating {$post['title']}: " . $e->getMessage() . "</p>";
            $errors[] = $post['title'] . ': ' . $e->getMessage();
        }
        
        usleep(100000); // Small delay for visual effect
    }
    
    echo "<hr>";
    echo "<h2 class='text-success'>✓ Blog Dummy Data Creation Complete!</h2>";
    echo "<p><strong>Successfully created:</strong> " . count($success) . " blog posts</p>";
    
    if (!empty($errors)) {
        echo "<p class='text-danger'><strong>Errors:</strong> " . count($errors) . "</p>";
        echo "<ul class='text-danger'>";
        foreach ($errors as $error) {
            echo "<li>{$error}</li>";
        }
        echo "</ul>";
    }
    
    echo "<div class='alert alert-info mt-4'>";
    echo "<h4>Next Steps:</h4>";
    echo "<ul>";
    echo "<li>Visit the <a href='../public/blog.php'>Blog Page</a> to see your posts</li>";
    echo "<li>Visit the <a href='../admin/cms-posts.php'>Admin Panel</a> to manage posts</li>";
    echo "<li>Visit the <a href='../index.php'>Homepage</a> to see latest posts</li>";
    echo "</ul>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div class='alert alert-danger'>";
    echo "<h3>Error!</h3>";
    echo "<p>" . $e->getMessage() . "</p>";
    echo "</div>";
}

echo "</div></div></body></html>";
?>
