<?php
/**
 * Enhanced Messages/Chat Interface with Real-time Updates
 * OBGYN Research & Collaboration Platform
 */

require_once '../config/config.php';

requireLogin();
if (!isResearcher()) {
    redirect(SITE_URL . '/index.php');
}

$pageTitle = 'Messages';
$userId = getCurrentUserId();

$isMobile = preg_match('/Mobile|Android|iPhone|iPad|iPod/', $_SERVER['HTTP_USER_AGENT']);
?>

<!-- Bootstrap CSS and JS for modal functionality -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>

<?php if (!$isMobile): ?>
<style>
/* Hide sidebar/menu on desktop for messages page */
.sidebar, .side-menu, #sidebar, .col-md-3, .col-lg-2 {
    display: none !important;
}
.main-content {
    margin-left: 0 !important;
    width: 100% !important;
}

/* Chat Header Styles */
.chat-header-clickable {
    cursor: pointer;
    transition: background-color 0.2s ease;
}

.chat-header-clickable:hover {
    background-color: #f8f9fa !important;
}

/* Participants Modal Styles */
.participant-item {
    display: flex;
    align-items: center;
    padding: 12px 16px;
    border-bottom: 1px solid #f0f2f5;
}

.participant-item:last-child {
    border-bottom: none;
}

.participant-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    margin-right: 12px;
    object-fit: cover;
}

.participant-avatar-placeholder {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    margin-right: 12px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    font-size: 14px;
}

.participant-info {
    flex: 1;
}

.participant-name {
    font-weight: 600;
    margin-bottom: 2px;
}

.participant-role {
    font-size: 12px;
    color: #6c757d;
}

.participant-actions {
    display: flex;
    gap: 8px;
}

/* Group Avatar Styles */
.group-avatar-container {
    position: relative;
    width: 40px;
    height: 40px;
    margin-right: 12px;
}

.group-avatar-item {
    width: 24px;
    height: 24px;
    border-radius: 50%;
    border: 2px solid white;
    object-fit: cover;
    position: absolute;
}

.group-avatar-item:nth-child(1) {
    top: 0;
    left: 0;
    z-index: 3;
}

.group-avatar-item:nth-child(2) {
    top: 0;
    right: 0;
    z-index: 2;
}

.group-avatar-item:nth-child(3) {
    bottom: 0;
    left: 50%;
    transform: translateX(-50%);
    z-index: 1;
}

.group-avatar-extra {
    position: absolute;
    bottom: 0;
    right: 0;
    background: #6c757d;
    color: white;
    border-radius: 50%;
    width: 20px;
    height: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 10px;
    font-weight: 600;
    border: 2px solid white;
    z-index: 4;
}

/* Conversation Avatar Styles */
.conversation-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    margin-right: 12px;
    object-fit: cover;
}

.conversation-avatar-placeholder {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    margin-right: 12px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    font-size: 14px;
}

/* Search Results */
.search-results {
    max-height: 200px;
    overflow-y: auto;
    border: 1px solid #dee2e6;
    border-radius: 0.375rem;
    background: white;
    margin-top: 8px;
}

.search-result-item {
    display: flex;
    align-items: center;
    padding: 10px 12px;
    cursor: pointer;
    border-bottom: 1px solid #f8f9fa;
    transition: background-color 0.2s;
}

.search-result-item:hover {
    background-color: #f8f9fa;
}

.search-result-item:last-child {
    border-bottom: none;
}

.search-result-avatar {
    width: 36px;
    height: 36px;
    border-radius: 50%;
    margin-right: 12px;
    object-fit: cover;
}

.search-result-avatar-placeholder {
    width: 36px;
    height: 36px;
    border-radius: 50%;
    margin-right: 12px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 12px;
}

.search-result-info h6 {
    margin: 0;
    font-size: 14px;
    font-weight: 600;
}

.search-result-info small {
    color: #6c757d;
    font-size: 12px;
}
</style>

<?php endif; ?>

<div class="container-fluid">
    <div class="row">
        <main class="<?php echo $isMobile ? 'col-md-9 ms-sm-auto col-lg-10' : 'col-12'; ?> px-md-4" style="<?php echo $isMobile ? 'padding-bottom: 100px;' : ''; ?>">
            <?php
            $userPhoto = '/assets/images/default-avatar.png';
            if ($userId) {
                $stmt = $db->prepare("SELECT photo FROM researcher_profiles WHERE user_id = ?");
                $stmt->execute([$userId]);
                $profileData = $stmt->fetch();
                $userPhoto = $profileData['photo'] ?? null;
                if ($userPhoto && !empty($userPhoto)) {
                    $userPhoto = UPLOADS_URL . '/profile_pictures/' . (strpos($userPhoto, 'assets/uploads/profile_pictures/') !== false ? str_replace('assets/uploads/profile_pictures/', '', $userPhoto) : $userPhoto);
                } else {
                    $userPhoto = '/assets/images/default-avatar.png';
                }
            }
            ?>
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">
                    <i class="fas fa-envelope"></i> Messages
                    <span id="total-unread-badge" class="badge bg-danger ms-2" style="display: none;">0</span>
                </h1>
                <div class="btn-toolbar">
                    <button id="new-chat-btn" class="btn btn-primary">
                        <i class="fas fa-plus"></i> New Message
                    </button>
                    <button id="refresh-btn" class="btn btn-outline-secondary ms-2">
                        <i class="fas fa-sync-alt"></i> Refresh
                    </button>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <div id="alert-container"></div>

            <div class="row" style="min-height: 70vh;">
                <!-- Chat List -->
                <div class="col-md-4 col-lg-3">
                    <div class="card shadow">
                        <div class="card-header">
                            <div class="input-group input-group-sm">
                                <input type="text" id="chat-search-input" class="form-control" placeholder="Search conversations...">
                                <button class="btn btn-outline-secondary" type="button">
                                    <i class="fas fa-search"></i>
                                </button>
                            </div>
                        </div>
                        <div id="chat-list" class="list-group list-group-flush" style="max-height: 600px; overflow-y: auto;">
                            <div class="text-center py-4">
                                <div class="spinner-border text-primary" role="status">
                                    <span class="visually-hidden">Loading...</span>
                                </div>
                                <p class="mt-3 small">Loading conversations...</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Chat Messages -->
                <div class="col-md-8 col-lg-9">
                    <div id="chat-container" class="card shadow">
                        <div class="card-header chat-header-clickable" id="chat-header" onclick="ChatManager.showParticipantsModal()">
                            <h6 class="mb-0 text-dark fw-bold">Select a conversation</h6>
                        </div>
                        <div id="messages-area" class="card-body text-center py-5" style="max-height: 400px; overflow-y: auto;">
                            <i class="fas fa-comments fa-4x text-muted mb-3"></i>
                            <h5>Select a conversation</h5>
                            <p class="text-muted">Choose a conversation from the list or start a new one</p>
                            <button id="start-new-chat-btn" class="btn btn-primary">
                                <i class="fas fa-plus"></i> Start New Conversation
                            </button>
                        </div>
                    </div>
                    <div id="send-box" class="mt-3" style="display: none;">
                        <!-- Send box will be populated here -->
                    </div>
                </div>
            </div>

            <script>
            // Chat Manager
            const ChatManager = {
                currentRoomId: null,
                currentRoomName: '',
                currentRoomParticipants: [],
                chatRooms: [],
                messages: [],
                lastMessageId: null,
                autoRefreshInterval: null,
                messageRefreshInterval: null,
                
                init() {
                    this.loadChatRooms();
                    this.setupEventListeners();
                    this.startAutoRefresh();
                },
                
                setupEventListeners() {
                    // New chat buttons
                    document.getElementById('new-chat-btn').addEventListener('click', () => {
                        this.showNewChatModal();
                    });
                    
                    document.getElementById('start-new-chat-btn').addEventListener('click', () => {
                        this.showNewChatModal();
                    });
                    
                    // Send new message
                    document.getElementById('send-new-message-btn').addEventListener('click', () => {
                        this.sendNewMessage();
                    });
                    
                    // Desktop new message search
                    const desktopNewMessageSearch = document.getElementById('desktop-new-message-search');
                    if (desktopNewMessageSearch) {
                        desktopNewMessageSearch.addEventListener('input', (e) => {
                            this.searchResearchersForNewMessage(e.target.value);
                        });
                    }
                    
                    // Refresh button
                    document.getElementById('refresh-btn').addEventListener('click', () => {
                        this.loadChatRooms();
                    });
                    
                    // Chat search
                    document.getElementById('chat-search-input').addEventListener('input', (e) => {
                        this.searchChats(e.target.value);
                    });
                    
                    // Enter key in message input
                    document.addEventListener('keydown', (e) => {
                        if (e.key === 'Enter' && !e.shiftKey) {
                            const messageInput = document.getElementById('message-input');
                            if (messageInput && messageInput === document.activeElement) {
                                e.preventDefault();
                                this.sendMessage();
                            }
                        }
                    });
                },
                
                async loadChatRooms() {
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/chat.php?action=get_rooms');
                        const data = await response.json();
                        
                        if (data.success) {
                            this.chatRooms = data.rooms || [];
                            this.renderChatList();
                            this.updateUnreadBadge();
                        } else {
                            this.showAlert('Error loading conversations', 'danger');
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        this.showAlert('Failed to load conversations', 'danger');
                    }
                },
                
                renderChatList() {
                    const container = document.getElementById('chat-list');
                    
                    if (this.chatRooms.length === 0) {
                        container.innerHTML = `
                            <div class="list-group-item text-center text-muted py-4">
                                <i class="fas fa-inbox fa-2x mb-2"></i>
                                <p class="mb-0 small">No conversations yet</p>
                                <button class="btn btn-sm btn-primary mt-2" onclick="ChatManager.showNewChatModal()">
                                    <i class="fas fa-plus"></i> Start One
                                </button>
                            </div>
                        `;
                        return;
                    }
                    
                    const html = this.chatRooms.map(room => {
                        const participantNames = room.participants.length > 1 ? 
                            room.participants.map(p => p.first_name + ' ' + p.last_name).join(', ') : 
                            (room.participants && room.participants.length > 0 ? 
                                room.participants[0].first_name + ' ' + room.participants[0].last_name : 
                                'Unknown');
                        
                        // Generate avatar HTML
                        let avatarHtml = '';
                        if (room.participants.length > 1) {
                            const displayParticipants = room.participants.slice(0, 3);
                            avatarHtml = `<div class="group-avatar-container">`;
                            displayParticipants.forEach((participant, index) => {
                                if (participant.photo) {
                                    avatarHtml += `<img src="${participant.photo}" class="group-avatar-item" alt="${this.escapeHtml(participant.first_name)}">`;
                                } else {
                                    const initials = this.getInitials(participant.first_name + ' ' + participant.last_name);
                                    avatarHtml += `<div class="group-avatar-item participant-avatar-placeholder" style="background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${initials}</div>`;
                                }
                            });
                            if (room.participants.length > 3) {
                                avatarHtml += `<div class="group-avatar-extra">+${room.participants.length - 3}</div>`;
                            }
                            avatarHtml += `</div>`;
                        } else {
                            const participant = room.participants && room.participants.length > 0 ? room.participants[0] : null;
                            if (participant && participant.photo) {
                                avatarHtml = `<img src="${participant.photo}" class="conversation-avatar" alt="${this.escapeHtml(participant.first_name)}">`;
                            } else {
                                const name = participant ? participant.first_name + ' ' + participant.last_name : 'Unknown';
                                const initials = this.getInitials(name);
                                avatarHtml = `<div class="conversation-avatar-placeholder" style="background: ${this.stringToColor(name)}">${initials}</div>`;
                            }
                        }
                        
                        return `
                        <a href="#" class="list-group-item list-group-item-action ${this.currentRoomId == room.id ? 'active' : ''}"
                           data-room-id="${room.id}"
                           onclick="ChatManager.selectRoom(${room.id}, '${this.escapeHtml(room.name || (room.participants.length > 1 ? 'Group Chat' : participantNames)).replace(/'/g, "\\'")}', event)">
                            <div class="d-flex align-items-center">
                                ${avatarHtml}
                                <div class="flex-grow-1" style="min-width: 0;">
                                    <div class="d-flex w-100 justify-content-between">
                                        <h6 class="mb-1 text-dark text-truncate">
                                            ${this.escapeHtml(room.name || (room.participants.length > 1 ? 'Group Chat' : participantNames))}
                                            ${room.unread_count > 0 ? `<span class="badge bg-danger ms-2">${room.unread_count}</span>` : ''}
                                        </h6>
                                        <small class="text-muted">${room.last_message_time ? this.timeAgo(room.last_message_time) : ''}</small>
                                    </div>
                                    <p class="mb-0 small text-truncate text-dark">
                                        ${this.escapeHtml(room.last_message || 'No messages')}
                                    </p>
                                    ${room.participants.length > 1 ? `<small class="text-muted">${room.participants.length} participants</small>` : ''}
                                </div>
                            </div>
                        </a>
                        `;
                    }).join('');
                    
                    container.innerHTML = html;
                },
                
                searchChats(query) {
                    const container = document.getElementById('chat-list');
                    const items = container.querySelectorAll('.list-group-item');
                    
                    if (!query.trim()) {
                        items.forEach(item => item.style.display = 'flex');
                        return;
                    }
                    
                    const searchTerm = query.toLowerCase();
                    items.forEach(item => {
                        const text = item.textContent.toLowerCase();
                        if (text.includes(searchTerm)) {
                            item.style.display = 'flex';
                        } else {
                            item.style.display = 'none';
                        }
                    });
                },
                
                updateUnreadBadge() {
                    const totalUnread = this.chatRooms.reduce((sum, room) => sum + (room.unread_count || 0), 0);
                    const badge = document.getElementById('total-unread-badge');
                    
                    if (totalUnread > 0) {
                        badge.textContent = totalUnread;
                        badge.style.display = 'inline-block';
                    } else {
                        badge.style.display = 'none';
                    }
                },
                
                async selectRoom(roomId, roomName, event) {
                    this.currentRoomId = roomId;
                    this.currentRoomName = roomName;
                    
                    // Load room details including participants
                    await this.loadRoomDetails();
                    
                    // Update chat header with participants
                    this.updateChatHeader();
                    
                    // Update active state
                    document.querySelectorAll('#chat-list .list-group-item').forEach(item => {
                        item.classList.remove('active');
                    });
                    if (event && event.target) {
                        event.target.closest('.list-group-item').classList.add('active');
                    } else {
                        // Fallback: find the item by room ID
                        const roomItem = document.querySelector(`#chat-list .list-group-item[data-room-id="${roomId}"]`);
                        if (roomItem) {
                            roomItem.classList.add('active');
                        }
                    }
                    
                    // Show skeleton loader
                    const messagesArea = document.getElementById('messages-area');
                    messagesArea.innerHTML = `
                        <div style="height: 400px; overflow-y: auto; padding: 1.5rem 1rem 1rem 1rem;" id="messageContainer">
                            <div class="skeleton-message mb-3">
                                <div class="skeleton skeleton-avatar"></div>
                                <div class="skeleton skeleton-text" style="width: 150px; height: 16px;"></div>
                                <div class="skeleton skeleton-text" style="width: 250px; height: 40px;"></div>
                                <div class="skeleton skeleton-text" style="width: 60px; height: 12px;"></div>
                            </div>
                            <div class="skeleton-message mb-3 text-end">
                                <div class="skeleton skeleton-text" style="width: 60px; height: 12px;"></div>
                                <div class="skeleton skeleton-text" style="width: 200px; height: 40px;"></div>
                            </div>
                            <div class="skeleton-message mb-3">
                                <div class="skeleton skeleton-avatar"></div>
                                <div class="skeleton skeleton-text" style="width: 120px; height: 16px;"></div>
                                <div class="skeleton skeleton-text" style="width: 180px; height: 40px;"></div>
                                <div class="skeleton skeleton-text" style="width: 60px; height: 12px;"></div>
                            </div>
                        </div>
                    `;
                    
                    // Load messages
                    await this.loadMessages();
                    
                    // Show send box
                    const sendBox = document.getElementById('send-box');
                    if (sendBox) {
                        sendBox.innerHTML = `
                            <div class="d-flex gap-2">
                                <input type="text" id="message-input" class="form-control" placeholder="Type your message..." maxlength="1000">
                                <button id="send-btn" class="btn btn-primary" onclick="ChatManager.sendMessage()">
                                    <i class="fas fa-paper-plane"></i> Send
                                </button>
                            </div>
                        `;
                        sendBox.style.display = 'block';
                        
                        // Focus input
                        document.getElementById('message-input').focus();
                    }
                    
                    // Start message auto-refresh
                    this.startMessageRefresh();
                },
                
                async loadRoomDetails() {
                    if (!this.currentRoomId) return;
                    
                    try {
                        const response = await fetch(`<?php echo SITE_URL; ?>/api/chat.php?action=get_room_details&room_id=${this.currentRoomId}`);
                        const data = await response.json();
                        
                        if (data.success) {
                            this.currentRoomParticipants = data.participants || [];
                        }
                    } catch (error) {
                        console.error('Error loading room details:', error);
                    }
                },
                
                updateChatHeader() {
                    const header = document.getElementById('chat-header');
                    if (!header) return;
                    
                    if (!this.currentRoomId) {
                        header.innerHTML = '<h6 class="mb-0 text-dark fw-bold">Select a conversation</h6>';
                        return;
                    }
                    
                    const isGroup = this.currentRoomParticipants.length > 1;
                    
                    if (isGroup) {
                        // Group chat header with multiple avatars
                        const displayParticipants = this.currentRoomParticipants.slice(0, 4);
                        let avatarsHtml = '<div class="group-avatar-container me-2">';
                        displayParticipants.forEach((participant, index) => {
                            if (participant.photo) {
                                avatarsHtml += `<img src="${participant.photo}" class="group-avatar-item" alt="${this.escapeHtml(participant.first_name)}">`;
                            } else {
                                const initials = this.getInitials(participant.first_name + ' ' + participant.last_name);
                                avatarsHtml += `<div class="group-avatar-item participant-avatar-placeholder" style="background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${initials}</div>`;
                            }
                        });
                        if (this.currentRoomParticipants.length > 4) {
                            avatarsHtml += `<div class="group-avatar-extra">+${this.currentRoomParticipants.length - 4}</div>`;
                        }
                        avatarsHtml += '</div>';
                        
                        header.innerHTML = `
                            <div class="d-flex justify-content-between align-items-center w-100">
                                <div class="d-flex align-items-center">
                                    ${avatarsHtml}
                                    <div>
                                        <h6 class="mb-0 text-dark fw-bold">${this.escapeHtml(this.currentRoomName)}</h6>
                                        <small class="text-muted">${this.currentRoomParticipants.length} participants</small>
                                    </div>
                                </div>
                                <div class="btn-group">
                                    <button class="btn btn-sm btn-outline-primary" onclick="event.stopPropagation(); ChatManager.showAddParticipantsModal()">
                                        <i class="fas fa-user-plus"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-secondary" onclick="event.stopPropagation(); ChatManager.showRoomInfo()">
                                        <i class="fas fa-info-circle"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-danger" onclick="event.stopPropagation(); ChatManager.leaveRoom()">
                                        <i class="fas fa-sign-out-alt"></i>
                                    </button>
                                </div>
                            </div>
                        `;
                    } else if (this.currentRoomParticipants.length > 0) {
                        // Direct message header
                        const participant = this.currentRoomParticipants[0] || {};
                        const avatarHtml = participant.photo ? 
                            `<img src="${participant.photo}" class="rounded-circle me-2" style="width: 32px; height: 32px;" alt="${this.escapeHtml(participant.first_name)}">` :
                            `<div class="rounded-circle participant-avatar-placeholder me-2" style="width: 32px; height: 32px; font-size: 12px; background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${this.getInitials(participant.first_name + ' ' + participant.last_name)}</div>`;
                        
                        header.innerHTML = `
                            <div class="d-flex justify-content-between align-items-center w-100">
                                <div class="d-flex align-items-center">
                                    ${avatarHtml}
                                    <h6 class="mb-0 text-dark fw-bold">${this.escapeHtml(this.currentRoomName)}</h6>
                                </div>
                                <div class="btn-group">
                                    <button class="btn btn-sm btn-outline-primary" onclick="event.stopPropagation(); ChatManager.showAddParticipantsModal()">
                                        <i class="fas fa-user-plus"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-secondary" onclick="event.stopPropagation(); ChatManager.showRoomInfo()">
                                        <i class="fas fa-info-circle"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-danger" onclick="event.stopPropagation(); ChatManager.leaveRoom()">
                                        <i class="fas fa-sign-out-alt"></i>
                                    </button>
                                </div>
                            </div>
                        `;
                    } else {
                        // Fallback when no participants loaded yet
                        header.innerHTML = `
                            <div class="d-flex justify-content-between align-items-center w-100">
                                <div class="d-flex align-items-center">
                                    <div class="rounded-circle participant-avatar-placeholder me-2" style="width: 32px; height: 32px; font-size: 12px; background: #6c757d">?</div>
                                    <h6 class="mb-0 text-dark fw-bold">${this.escapeHtml(this.currentRoomName)}</h6>
                                </div>
                                <div class="btn-group">
                                    <button class="btn btn-sm btn-outline-primary" onclick="event.stopPropagation(); ChatManager.showAddParticipantsModal()">
                                        <i class="fas fa-user-plus"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-secondary" onclick="event.stopPropagation(); ChatManager.showRoomInfo()">
                                        <i class="fas fa-info-circle"></i>
                                    </button>
                                    <button class="btn btn-sm btn-outline-danger" onclick="event.stopPropagation(); ChatManager.leaveRoom()">
                                        <i class="fas fa-sign-out-alt"></i>
                                    </button>
                                </div>
                            </div>
                        `;
                    }
                },
                
                showParticipantsModal() {
                    if (!this.currentRoomId) return;
                    
                    const modal = new bootstrap.Modal(document.getElementById('participantsModal'));
                    
                    // Populate participants list
                    const participantsList = document.getElementById('participants-list');
                    participantsList.innerHTML = this.currentRoomParticipants.map(participant => {
                        const isCurrentUser = participant.id == <?php echo $userId; ?>;
                        const avatarHtml = participant.photo ? 
                            `<img src="${participant.photo}" class="participant-avatar" alt="${this.escapeHtml(participant.first_name)}">` :
                            `<div class="participant-avatar-placeholder" style="background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${this.getInitials(participant.first_name + ' ' + participant.last_name)}</div>`;
                        
                        return `
                            <div class="participant-item">
                                ${avatarHtml}
                                <div class="participant-info">
                                    <div class="participant-name">${this.escapeHtml(participant.first_name + ' ' + participant.last_name)}</div>
                                    <div class="participant-role">${participant.role} ${isCurrentUser ? '(You)' : ''}</div>
                                </div>
                                ${!isCurrentUser ? `
                                <div class="participant-actions">
                                    ${participant.role !== 'admin' ? `
                                    <button class="btn btn-sm btn-success" onclick="ChatManager.makeAdmin(${participant.id})" title="Make Admin">
                                        <i class="fas fa-user-shield"></i>
                                    </button>
                                    ` : ''}
                                    <button class="btn btn-sm btn-danger" onclick="ChatManager.removeParticipant(${participant.id}, '${this.escapeHtml(participant.first_name + ' ' + participant.last_name).replace(/'/g, "\\'")}')" title="Remove">
                                        <i class="fas fa-user-minus"></i>
                                    </button>
                                </div>
                                ` : ''}
                            </div>
                        `;
                    }).join('');
                    
                    modal.show();
                },
                
                async makeAdmin(userId) {
                    if (!this.currentRoomId) return;
                    
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/chat.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                action: 'change_role',
                                room_id: this.currentRoomId,
                                user_id: userId,
                                role: 'admin'
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            // Update local data
                            this.currentRoomParticipants.forEach(p => {
                                if (p.id === userId) {
                                    p.role = 'admin';
                                }
                            });
                            
                            // Update the modal content without closing/reopening it
                            const participantsList = document.getElementById('participants-list');
                            if (participantsList) {
                                participantsList.innerHTML = this.currentRoomParticipants.map(participant => {
                                    const isCurrentUser = participant.id == <?php echo $userId; ?>;
                                    const avatarHtml = participant.photo ? 
                                        `<img src="${participant.photo}" class="participant-avatar" alt="${this.escapeHtml(participant.first_name)}">` :
                                        `<div class="participant-avatar-placeholder" style="background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${this.getInitials(participant.first_name + ' ' + participant.last_name)}</div>`;
                                    
                                    return `
                                        <div class="participant-item">
                                            ${avatarHtml}
                                            <div class="participant-info">
                                                <div class="participant-name">${this.escapeHtml(participant.first_name + ' ' + participant.last_name)}</div>
                                                <div class="participant-role">${participant.role} ${isCurrentUser ? '(You)' : ''}</div>
                                            </div>
                                            ${!isCurrentUser ? `
                                            <div class="participant-actions">
                                                ${participant.role !== 'admin' ? `
                                                <button class="btn btn-sm btn-success" onclick="ChatManager.makeAdmin(${participant.id})" title="Make Admin">
                                                    <i class="fas fa-user-shield"></i>
                                                </button>
                                                ` : ''}
                                                <button class="btn btn-sm btn-danger" onclick="ChatManager.removeParticipant(${participant.id}, '${this.escapeHtml(participant.first_name + ' ' + participant.last_name).replace(/'/g, "\\'")}')" title="Remove">
                                                    <i class="fas fa-user-minus"></i>
                                                </button>
                                            </div>
                                            ` : ''}
                                        </div>
                                    `;
                                }).join('');
                            }
                            
                            this.showAlert('User role updated successfully', 'success');
                        } else {
                            this.showAlert(data.message || 'Failed to update role', 'danger');
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        this.showAlert('Failed to update role', 'danger');
                    }
                },
                
                async removeParticipant(userId, userName) {
                    if (!this.currentRoomId || !confirm(`Are you sure you want to remove ${userName} from this chat?`)) {
                        return;
                    }
                    
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/chat.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                action: 'remove_participant',
                                room_id: this.currentRoomId,
                                user_id: userId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            // Send system message
                            await this.sendSystemMessage(`[SYSTEM] ${userName}|${userId} was removed from the conversation`);
                            
                            // Update local data
                            this.currentRoomParticipants = this.currentRoomParticipants.filter(p => p.id !== userId);
                            
                            // Update the modal content without closing/reopening it
                            const participantsList = document.getElementById('participants-list');
                            if (participantsList) {
                                participantsList.innerHTML = this.currentRoomParticipants.map(participant => {
                                    const isCurrentUser = participant.id == <?php echo $userId; ?>;
                                    const avatarHtml = participant.photo ? 
                                        `<img src="${participant.photo}" class="participant-avatar" alt="${this.escapeHtml(participant.first_name)}">` :
                                        `<div class="participant-avatar-placeholder" style="background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${this.getInitials(participant.first_name + ' ' + participant.last_name)}</div>`;
                                    
                                    return `
                                        <div class="participant-item">
                                            ${avatarHtml}
                                            <div class="participant-info">
                                                <div class="participant-name">${this.escapeHtml(participant.first_name + ' ' + participant.last_name)}</div>
                                                <div class="participant-role">${participant.role} ${isCurrentUser ? '(You)' : ''}</div>
                                            </div>
                                            ${!isCurrentUser ? `
                                            <div class="participant-actions">
                                                ${participant.role !== 'admin' ? `
                                                <button class="btn btn-sm btn-success" onclick="ChatManager.makeAdmin(${participant.id})" title="Make Admin">
                                                    <i class="fas fa-user-shield"></i>
                                                </button>
                                                ` : ''}
                                                <button class="btn btn-sm btn-danger" onclick="ChatManager.removeParticipant(${participant.id}, '${this.escapeHtml(participant.first_name + ' ' + participant.last_name).replace(/'/g, "\\'")}')" title="Remove">
                                                    <i class="fas fa-user-minus"></i>
                                                </button>
                                            </div>
                                            ` : ''}
                                        </div>
                                    `;
                                }).join('');
                            }
                            
                            // Reload chat rooms to reflect changes
                            await this.loadChatRooms();
                            
                            // If we removed ourselves, close the chat
                            if (userId == <?php echo $userId; ?>) {
                                this.closeChat();
                                // Close the modal
                                const modal = bootstrap.Modal.getInstance(document.getElementById('participantsModal'));
                                if (modal) modal.hide();
                            }
                        } else {
                            this.showAlert(data.message || 'Failed to remove participant', 'danger');
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        this.showAlert('Failed to remove participant', 'danger');
                    }
                },
                
                async loadMessages(append = false) {
                    if (!this.currentRoomId) return;
                    
                    try {
                        const since = append && this.lastMessageId ? `&since_id=${this.lastMessageId}` : '';
                        const response = await fetch(`<?php echo SITE_URL; ?>/api/chat.php?action=get_messages&room_id=${this.currentRoomId}&limit=100${since}`);
                        const data = await response.json();
                        
                        if (data.success) {
                            const newMessages = data.messages || [];
                            
                            if (append) {
                                // Append new messages
                                this.messages.push(...newMessages);
                                this.messages.sort((a, b) => a.id - b.id);
                                this.appendMessages(newMessages.sort((a, b) => a.id - b.id));
                            } else {
                                // Replace all messages
                                this.messages = newMessages;
                                this.renderChat();
                            }
                            
                            // Update last message ID
                            if (this.messages.length > 0) {
                                this.lastMessageId = Math.max(...this.messages.map(m => m.id));
                            }
                            
                            // Mark as read
                            await this.markRoomAsRead();
                        } else {
                            this.showAlert('Error loading messages', 'danger');
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        this.showAlert('Failed to load messages', 'danger');
                    }
                },
                
                appendMessages(newMessages) {
                    if (newMessages.length === 0) return;
                    
                    const messageContainer = document.getElementById('messageContainer');
                    if (!messageContainer) return;
                    
                    const fragment = document.createDocumentFragment();
                    
                    newMessages.forEach(msg => {
                        const messageElement = this.createMessageElement(msg);
                        fragment.appendChild(messageElement);
                    });
                    
                    messageContainer.appendChild(fragment);
                    
                    // Scroll to bottom
                    this.scrollToBottom();
                },
                
                createMessageElement(msg) {
                    // Handle system messages
                    if (msg.is_system || msg.sender_id == 0 || msg.sender_id == null || msg.message.startsWith('[SYSTEM]')) {
                        const cleanMessage = msg.message.replace('[SYSTEM] ', '');
                        const div = document.createElement('div');
                        div.className = 'mb-3 text-center';
                        div.innerHTML = `
                            <div class="d-inline-block bg-light text-muted px-3 py-2 rounded-pill small">
                                <i class="fas fa-info-circle me-1"></i>
                                ${this.processSystemMessage(cleanMessage)}
                            </div>
                        `;
                        return div;
                    }
                    
                    const isSent = msg.sender_id == <?php echo $userId; ?>;
                    
                    const div = document.createElement('div');
                    div.className = 'mb-3 message-new';
                    
                    if (isSent) {
                        div.classList.add('text-end');
                        div.innerHTML = `
                            <div class="d-inline-block" style="max-width: 70%;">
                                <div class="d-flex align-items-end justify-content-end">
                                    <small class="text-muted me-2">${this.formatTime(msg.created_at)}</small>
                                    <div class="bg-primary text-white p-3 rounded">
                                        ${this.escapeHtml(msg.message).replace(/\n/g, '<br>')}
                                    </div>
                                </div>
                            </div>
                        `;
                    } else {
                        const avatarHtml = msg.sender_photo ? 
                            `<img src="${msg.sender_photo}" class="rounded-circle me-2" style="width: 32px; height: 32px;" alt="${this.escapeHtml(msg.sender_name)}">` :
                            `<div class="rounded-circle bg-secondary text-white d-flex align-items-center justify-content-center me-2" style="width: 32px; height: 32px; font-size: 12px; font-weight: bold; background: ${this.stringToColor(msg.sender_name)}">${this.getInitials(msg.sender_name)}</div>`;
                        
                        div.innerHTML = `
                            <div class="d-flex align-items-start">
                                ${avatarHtml}
                                <div class="flex-grow-1">
                                    <small class="text-muted d-block mb-1">
                                        ${this.escapeHtml(msg.sender_name)}
                                    </small>
                                    <div class="d-flex align-items-end">
                                        <div class="bg-light p-3 rounded flex-grow-1 me-2">
                                            ${this.escapeHtml(msg.message).replace(/\n/g, '<br>')}
                                        </div>
                                        <small class="text-muted">${this.formatTime(msg.created_at)}</small>
                                    </div>
                                </div>
                            </div>
                        `;
                    }
                    
                    return div;
                },
                
                renderChat() {
                    const messagesArea = document.getElementById('messages-area');
                    const sendBox = document.getElementById('send-box');
                    
                    if (!this.currentRoomId) {
                        messagesArea.innerHTML = `
                            <div class="text-center py-5">
                                <i class="fas fa-comments fa-4x text-muted mb-3"></i>
                                <h5>Select a conversation</h5>
                                <p class="text-muted">Choose a conversation from the list or start a new one</p>
                                <button class="btn btn-primary" onclick="ChatManager.showNewChatModal()">
                                    <i class="fas fa-plus"></i> Start New Conversation
                                </button>
                            </div>
                        `;
                        if (sendBox) sendBox.style.display = 'none';
                        return;
                    }
                    
                    const messagesHtml = this.messages.length === 0 ? `
                        <div class="text-center text-muted py-5">
                            <i class="fas fa-comments fa-3x mb-3"></i>
                            <p>No messages yet. Start the conversation!</p>
                        </div>
                    ` : this.messages.map(msg => {
                        // Handle system messages
                        if (msg.is_system || msg.sender_id == 0 || msg.sender_id == null || msg.message.startsWith('[SYSTEM]')) {
                            const cleanMessage = msg.message.replace('[SYSTEM] ', '');
                            return `
                                <div class="mb-3 text-center">
                                    <div class="d-inline-block bg-light text-muted px-3 py-2 rounded-pill small">
                                        <i class="fas fa-info-circle me-1"></i>
                                        ${this.processSystemMessage(cleanMessage)}
                                    </div>
                                </div>
                            `;
                        }
                        
                        const isSent = msg.sender_id == <?php echo $userId; ?>;
                        if (isSent) {
                            return `
                                <div class="d-flex justify-content-end mb-3">
                                    <div class="d-inline-block" style="max-width: 70%;">
                                        <div class="d-flex align-items-end justify-content-end">
                                            <small class="text-muted me-2">${this.formatTime(msg.created_at)}</small>
                                            <div class="bg-primary text-white p-3 rounded">
                                                ${this.escapeHtml(msg.message).replace(/\n/g, '<br>')}
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            `;
                        } else {
                            const avatarHtml = msg.sender_photo ? 
                                `<img src="${msg.sender_photo}" class="rounded-circle me-2" style="width: 32px; height: 32px;" alt="${this.escapeHtml(msg.sender_name)}">` :
                                `<div class="rounded-circle bg-secondary text-white d-flex align-items-center justify-content-center me-2" style="width: 32px; height: 32px; font-size: 12px; font-weight: bold; background: ${this.stringToColor(msg.sender_name)}">${this.getInitials(msg.sender_name)}</div>`;
                            
                            return `
                                <div class="d-flex align-items-start mb-3">
                                    ${avatarHtml}
                                    <div class="flex-grow-1">
                                        <small class="text-muted d-block mb-1">
                                            ${this.escapeHtml(msg.sender_name)}
                                        </small>
                                        <div class="d-flex align-items-end">
                                            <div class="bg-light p-3 rounded flex-grow-1 me-2">
                                                ${this.escapeHtml(msg.message).replace(/\n/g, '<br>')}
                                            </div>
                                            <small class="text-muted">${this.formatTime(msg.created_at)}</small>
                                        </div>
                                    </div>
                                </div>
                            `;
                        }
                    }).join('');
                    
                    messagesArea.innerHTML = `
                        <div style="height: 400px; overflow-y: auto; padding: 1.5rem 1rem 1rem 1rem;" id="messageContainer">
                            ${messagesHtml}
                        </div>
                    `;
                    
                    // Scroll to bottom
                    this.scrollToBottom();
                },
                
                async sendMessage() {
                    const input = document.getElementById('message-input');
                    const message = input.value.trim();
                    
                    if (!message || !this.currentRoomId) return;
                    
                    // Clear input immediately for better UX
                    input.value = '';
                    
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/chat.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                action: 'send_message',
                                room_id: this.currentRoomId,
                                message: message
                            })
                        });
                        
                        if (!response.ok) {
                            throw new Error(`HTTP error! status: ${response.status}`);
                        }
                        
                        let data;
                        try {
                            data = await response.json();
                        } catch (parseError) {
                            console.error('JSON parse error:', parseError);
                            throw new Error('Invalid response format');
                        }
                        
                        if (data.success) {
                            // Message sent successfully, refresh messages to show the new one
                            await this.loadMessages(true);
                            
                            // Scroll to bottom
                            setTimeout(() => this.scrollToBottom(), 100);
                        } else {
                            // Restore the message if sending failed
                            input.value = message;
                            this.showAlert('Failed to send message', 'danger');
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        // Restore the message if sending failed
                        input.value = message;
                        this.showAlert('Failed to send message', 'danger');
                    }
                },
                
                async markRoomAsRead() {
                    if (!this.currentRoomId) return;
                    
                    try {
                        await fetch('<?php echo SITE_URL; ?>/api/chat.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                action: 'mark_read',
                                room_id: this.currentRoomId
                            })
                        });
                        
                        // Update local data
                        this.chatRooms.forEach(room => {
                            if (room.id == this.currentRoomId) {
                                room.unread_count = 0;
                            }
                        });
                        this.renderChatList();
                        this.updateUnreadBadge();
                    } catch (error) {
                        console.error('Error marking as read:', error);
                    }
                },
                
                showNewChatModal() {
                    const modal = new bootstrap.Modal(document.getElementById('newChatModal'));
                    modal.show();
                    this.loadResearchersForNewMessage();
                },
                
                async loadResearchersForNewMessage() {
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/search.php?q=&type=researchers&limit=100');
                        const data = await response.json();
                        
                        const container = document.getElementById('desktop-new-message-results');
                        if (data.success && data.results) {
                            // Filter out current user
                            const currentUserId = <?php echo $userId; ?>;
                            const availableUsers = data.results.filter(user => user.id != currentUserId);
                            
                            if (availableUsers.length === 0) {
                                container.innerHTML = '<div class="text-center text-muted py-3">No researchers available</div>';
                            } else {
                                container.innerHTML = availableUsers.map(user => {
                                    const letterAvatar = `<div class="search-result-avatar-placeholder" style="background: ${this.stringToColor(user.first_name + ' ' + user.last_name)}">${this.getInitials(user.first_name + ' ' + user.last_name)}</div>`;
                                    const avatarHtml = user.photo ? 
                                        `<img src="${user.photo}" class="search-result-avatar" alt="${this.escapeHtml(user.first_name)}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"><div class="search-result-avatar-placeholder" style="background: ${this.stringToColor(user.first_name + ' ' + user.last_name)}; display: none;">${this.getInitials(user.first_name + ' ' + user.last_name)}</div>` :
                                        letterAvatar;
                                    
                                    return `
                                        <div class="search-result-item" onclick="ChatManager.selectResearcherForNewMessage(${user.id}, '${this.escapeHtml(user.first_name + ' ' + user.last_name).replace(/'/g, "\\'")}')">
                                            ${avatarHtml}
                                            <div class="search-result-info">
                                                <h6 class="mb-0">${this.escapeHtml(user.first_name + ' ' + user.last_name)}</h6>
                                                <small class="text-muted">${user.institution || ''}</small>
                                            </div>
                                        </div>
                                    `;
                                }).join('');
                            }
                        } else {
                            container.innerHTML = '<div class="text-center text-muted py-3">Error loading researchers</div>';
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        document.getElementById('desktop-new-message-results').innerHTML = '<div class="text-center text-muted py-3">Error loading researchers</div>';
                    }
                },
                
                showAddParticipantsModal() {
                    if (!this.currentRoomId) return;
                    const modal = new bootstrap.Modal(document.getElementById('addParticipantsModal'));
                    modal.show();
                    this.loadResearchersForAdding();
                },
                
                async loadResearchersForAdding() {
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/search.php?q=&type=researchers&limit=100');
                        const data = await response.json();
                        
                        const container = document.getElementById('add-participant-results');
                        if (data.success && data.results) {
                            // Filter out users already in the current room
                            const currentParticipants = this.chatRooms.find(room => room.id == this.currentRoomId)?.participants || [];
                            const currentUserIds = currentParticipants.map(p => p.id);
                            
                            const availableUsers = data.results.filter(user => !currentUserIds.includes(user.id));
                            
                            if (availableUsers.length === 0) {
                                container.innerHTML = '<div class="text-center text-muted py-3">No researchers available to add</div>';
                            } else {
                                container.innerHTML = availableUsers.map(user => {
                                    const letterAvatar = `<div class="search-result-avatar-placeholder" style="background: ${this.stringToColor(user.first_name + ' ' + user.last_name)}">${this.getInitials(user.first_name + ' ' + user.last_name)}</div>`;
                                    const avatarHtml = user.photo ? 
                                        `<img src="${user.photo}" class="search-result-avatar" alt="${this.escapeHtml(user.first_name)}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"><div class="search-result-avatar-placeholder" style="background: ${this.stringToColor(user.first_name + ' ' + user.last_name)}; display: none;">${this.getInitials(user.first_name + ' ' + user.last_name)}</div>` :
                                        letterAvatar;
                                    
                                    return `
                                        <div class="search-result-item" onclick="ChatManager.addParticipant(${user.id})">
                                            ${avatarHtml}
                                            <div class="search-result-info">
                                                <h6 class="mb-0">${this.escapeHtml(user.first_name + ' ' + user.last_name)}</h6>
                                                <small class="text-muted">${user.institution || ''}</small>
                                            </div>
                                        </div>
                                    `;
                                }).join('');
                            }
                        } else {
                            container.innerHTML = '<div class="text-center text-muted py-3">Error loading researchers</div>';
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        document.getElementById('add-participant-results').innerHTML = '<div class="text-center text-muted py-3">Error loading researchers</div>';
                    }
                },
                
                async addParticipant(userId) {
                    if (!this.currentRoomId) return;
                    
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/chat.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                action: 'add_participant',
                                room_id: this.currentRoomId,
                                user_id: userId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            // Send system message about the new participant
                            await this.sendSystemMessage(`[SYSTEM] ${data.participant_name}|${userId} was added to the conversation`);
                            
                            bootstrap.Modal.getInstance(document.getElementById('addParticipantsModal')).hide();
                            // Remove any remaining modal backdrops
                            document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
                            
                            // Refresh room details and chat list
                            await this.loadRoomDetails();
                            await this.loadChatRooms();
                            this.updateChatHeader();
                            
                            // No alert message - silent addition
                        } else {
                            this.showAlert(data.message || 'Failed to add participant', 'danger');
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        this.showAlert('Failed to add participant', 'danger');
                    }
                },
                
                async sendSystemMessage(message) {
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/chat.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                action: 'send_system_message',
                                room_id: this.currentRoomId,
                                message: message
                            })
                        });
                        
                        const data = await response.json();
                        if (data.success) {
                            // Immediately refresh messages to show the system message
                            await this.loadMessages(true);
                        } else {
                            console.error('Failed to send system message:', data.message);
                        }
                    } catch (error) {
                        console.error('Error sending system message:', error);
                    }
                },
                
                async sendNewMessage() {
                    const message = document.getElementById('desktop-initial-message').value.trim();
                    
                    if (!this.selectedUserForNewMessage || !message) {
                        this.showAlert('Please select a researcher and enter a message', 'warning');
                        return;
                    }
                    
                    try {
                        const response = await fetch('<?php echo SITE_URL; ?>/api/chat.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                action: 'start_conversation',
                                user_id: this.selectedUserForNewMessage.id,
                                message: message
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            const modal = bootstrap.Modal.getInstance(document.getElementById('newChatModal'));
                            if (modal) {
                                modal.hide();
                            }
                            // Remove any remaining modal backdrops
                            document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
                            document.getElementById('desktop-initial-message').value = '';
                            this.selectedUserForNewMessage = null;
                            this.showAlert('Message sent successfully!', 'success');
                            this.loadChatRooms(); // Refresh chat list
                        } else {
                            this.showAlert(data.message || 'Failed to send message', 'danger');
                        }
                    } catch (error) {
                        console.error('Error:', error);
                        this.showAlert('Failed to send message', 'danger');
                    }
                },
                
                selectResearcherForNewMessage(userId, userName) {
                    // Highlight selected researcher
                    const items = document.querySelectorAll('#desktop-new-message-results .search-result-item');
                    items.forEach(item => item.classList.remove('selected'));
                    event.currentTarget.classList.add('selected');
                    
                    // Store selected user
                    this.selectedUserForNewMessage = { id: userId, name: userName };
                },
                
                searchResearchersForNewMessage(query) {
                    const container = document.getElementById('desktop-new-message-results');
                    const items = container.querySelectorAll('.search-result-item');
                    
                    if (!query.trim()) {
                        // Show all when search is empty
                        this.loadResearchersForNewMessage();
                        return;
                    }
                    
                    const searchTerm = query.toLowerCase();
                    let hasVisibleItems = false;
                    
                    items.forEach(item => {
                        const text = item.textContent.toLowerCase();
                        if (text.includes(searchTerm)) {
                            item.style.display = 'flex';
                            hasVisibleItems = true;
                        } else {
                            item.style.display = 'none';
                        }
                    });
                    
                    if (!hasVisibleItems) {
                        container.innerHTML = '<div class="text-center text-muted py-3">No researchers found</div>';
                    }
                },
                
                startAutoRefresh() {
                    // Refresh chat list every 30 seconds
                    this.autoRefreshInterval = setInterval(() => {
                        this.loadChatRooms();
                    }, 30000);
                },
                
                startMessageRefresh() {
                    // Stop previous interval
                    if (this.messageRefreshInterval) {
                        clearInterval(this.messageRefreshInterval);
                    }
                    
                    // Refresh messages every 10 seconds when a room is selected
                    this.messageRefreshInterval = setInterval(() => {
                        if (this.currentRoomId) {
                            this.loadMessages(true); // Append new messages
                        }
                    }, 10000);
                },
                
                closeChat() {
                    this.currentRoomId = null;
                    this.currentRoomName = '';
                    this.currentRoomParticipants = [];
                    this.messages = [];
                    this.renderChat();
                    this.renderChatList();
                    
                    // Hide send box
                    const sendBox = document.getElementById('send-box');
                    if (sendBox) sendBox.style.display = 'none';
                },
                
                scrollToBottom() {
                    const container = document.getElementById('messageContainer');
                    if (container) {
                        container.scrollTop = container.scrollHeight;
                    }
                },
                
                processSystemMessage(message) {
                    // Handle system messages like "John Doe|123 was added to the conversation"
                    if (message.includes('was added to the conversation')) {
                        const nameWithId = message.replace(' was added to the conversation', '');
                        const [name, id] = nameWithId.split('|');
                        if (id) {
                            return `<i class="fas fa-user-plus text-success me-1"></i><a href="<?php echo SITE_URL; ?>/public/researcher-profile.php?id=${id}" class="text-decoration-none fw-bold" style="color: inherit;">${this.escapeHtml(name)}</a> was added to the conversation`;
                        } else {
                            return `<i class="fas fa-user-plus text-success me-1"></i><span class="fw-bold">${this.escapeHtml(name)}</span> was added to the conversation`;
                        }
                    }
                    if (message.includes('was removed from the conversation')) {
                        const nameWithId = message.replace(' was removed from the conversation', '');
                        const [name, id] = nameWithId.split('|');
                        if (id) {
                            return `<i class="fas fa-user-minus text-danger me-1"></i><a href="<?php echo SITE_URL; ?>/public/researcher-profile.php?id=${id}" class="text-decoration-none fw-bold" style="color: inherit;">${this.escapeHtml(name)}</a> was removed from the conversation`;
                        } else {
                            return `<i class="fas fa-user-minus text-danger me-1"></i><span class="fw-bold">${this.escapeHtml(name)}</span> was removed from the conversation`;
                        }
                    }
                    return this.escapeHtml(message);
                },
                
                showAlert(message, type = 'info') {
                    const container = document.getElementById('alert-container');
                    const alert = document.createElement('div');
                    alert.className = `alert alert-${type} alert-dismissible fade show`;
                    alert.innerHTML = `
                        ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    `;
                    container.appendChild(alert);
                    
                    // Auto-dismiss after 3 seconds
                    setTimeout(() => {
                        alert.remove();
                    }, 3000);
                },
                
                escapeHtml(text) {
                    const div = document.createElement('div');
                    div.textContent = text;
                    return div.innerHTML;
                },
                
                getInitials(name) {
                    if (!name) return '?';
                    const parts = name.trim().split(' ');
                    if (parts.length >= 2) {
                        return (parts[0][0] + parts[1][0]).toUpperCase();
                    } else if (parts.length === 1) {
                        return parts[0][0].toUpperCase();
                    }
                    return '?';
                },
                
                stringToColor(str) {
                    let hash = 0;
                    for (let i = 0; i < str.length; i++) {
                        hash = str.charCodeAt(i) + ((hash << 5) - hash);
                    }
                    let color = '#';
                    for (let i = 0; i < 3; i++) {
                        const value = (hash >> (i * 8)) & 0xFF;
                        color += ('00' + value.toString(16)).substr(-2);
                    }
                    return color;
                },
                
                timeAgo(dateString) {
                    const date = new Date(dateString);
                    const now = new Date();
                    const seconds = Math.floor((now - date) / 1000);
                    
                    const intervals = {
                        year: 31536000,
                        month: 2592000,
                        week: 604800,
                        day: 86400,
                        hour: 3600,
                        minute: 60
                    };
                    
                    for (const [unit, secondsInUnit] of Object.entries(intervals)) {
                        const interval = Math.floor(seconds / secondsInUnit);
                        if (interval >= 1) {
                            return interval === 1 ? `1 ${unit} ago` : `${interval} ${unit}s ago`;
                        }
                    }
                    
                    return 'Just now';
                },
                
                formatTime(dateString) {
                    const date = new Date(dateString);
                    return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                },
                
                showRoomInfo() {
                    // Placeholder for room info modal
                    this.showAlert('Room info feature coming soon', 'info');
                },
                
                leaveRoom() {
                    if (confirm('Are you sure you want to leave this conversation?')) {
                        this.closeChat();
                        this.showAlert('You have left the conversation', 'info');
                    }
                }
            };

            // Initialize on page load
            document.addEventListener('DOMContentLoaded', () => {
                ChatManager.init();
            });

            // Cleanup on page unload
            window.addEventListener('beforeunload', () => {
                if (ChatManager.autoRefreshInterval) {
                    clearInterval(ChatManager.autoRefreshInterval);
                }
                if (ChatManager.messageRefreshInterval) {
                    clearInterval(ChatManager.messageRefreshInterval);
                }
            });
            </script>

            <style>
            .list-group-item {
                transition: background-color 0.3s ease;
            }

            .list-group-item:hover {
                background-color: #f8f9fa !important;
            }

            .list-group-item.active {
                background-color: #007bff !important;
                border-color: #007bff;
            }

            #messageContainer {
                scroll-behavior: smooth;
            }

            #refresh-btn .fa-sync-alt {
                transition: transform 0.3s ease;
            }

            #refresh-btn:active .fa-sync-alt {
                transform: rotate(360deg);
            }

            #message-input {
                resize: none;
            }

            /* Skeleton Loading */
            .skeleton {
                background: linear-gradient(90deg, #f0f0f0 25%, #e0e0e0 50%, #f0f0f0 75%);
                background-size: 200% 100%;
                animation: skeleton-loading 1.5s infinite;
                border-radius: 4px;
            }

            .skeleton-avatar {
                width: 40px;
                height: 40px;
                border-radius: 50%;
            }

            .skeleton-text {
                height: 16px;
                margin-bottom: 8px;
            }

            .skeleton-button {
                border-radius: 4px;
            }

            @keyframes skeleton-loading {
                0% {
                    background-position: 200% 0;
                }
                100% {
                    background-position: -200% 0;
                }
            }

            /* Message Animation */
            .message-new {
                animation: message-slide-in 0.5s ease-out;
                opacity: 0;
                animation-fill-mode: forwards;
            }

            @keyframes message-slide-in {
                0% {
                    opacity: 0;
                    transform: translateY(20px);
                }
                100% {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            </style>
        </main>
    </div>
</div>

<!-- New Chat Modal -->
<div class="modal fade" id="newChatModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">New Message</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">Search Researcher</label>
                    <input type="text" id="desktop-new-message-search" class="form-control" placeholder="Type to search researchers...">
                </div>
                <div id="desktop-new-message-results" class="search-results">
                    <div class="text-center text-muted py-3">Loading researchers...</div>
                </div>
                <div class="mb-3">
                    <label class="form-label">Message</label>
                    <textarea id="desktop-initial-message" class="form-control" rows="3" placeholder="Type your first message..."></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" onclick="document.querySelectorAll('.modal-backdrop').forEach(b => b.remove())">Cancel</button>
                <button id="send-new-message-btn" type="button" class="btn btn-primary">Send Message</button>
            </div>
        </div>
    </div>
</div>

<!-- Add Participants Modal -->
<div class="modal fade" id="addParticipantsModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add Participants</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">Search Researchers</label>
                    <input type="text" id="add-participant-search" class="form-control" placeholder="Type to search researchers...">
                </div>
                <div id="add-participant-results" class="search-results">
                    <div class="text-center text-muted py-3">Loading researchers...</div>
                </div>
                <div class="alert alert-info mt-3">
                    <i class="fas fa-info-circle"></i> Adding a participant will convert this conversation to a group chat if it isn't already one.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" onclick="document.querySelectorAll('.modal-backdrop').forEach(b => b.remove())">Cancel</button>
            </div>
        </div>
    </div>
</div>

<!-- Participants Modal -->
<div class="modal fade" id="participantsModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Chat Participants</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body p-0">
                <div id="participants-list">
                    <!-- Participants will be populated here -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
// Add search functionality for adding participants
document.getElementById('add-participant-search').addEventListener('input', function(e) {
    const searchTerm = e.target.value.toLowerCase();
    const results = document.getElementById('add-participant-results');
    const items = results.querySelectorAll('.search-result-item');
    
    items.forEach(item => {
        const text = item.textContent.toLowerCase();
        if (text.includes(searchTerm)) {
            item.style.display = 'flex';
        } else {
            item.style.display = 'none';
        }
    });
});

// Global function for mobile compatibility
function showNewChatModal() {
    ChatManager.showNewChatModal();
}
</script>

<?php
// Include appropriate footer based on device
if ($isMobile) {
    include '../templates/researcher-mobile-footer.php';
} else {
    include '../templates/footer.php';
}
?>